/*=========================================================================

  Module:    $RCSfile: V_PyramidMetric.cpp,v $

  Copyright (c) 2006 Sandia Corporation.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

/*
 *
 * PyramidMetrics.cpp contains quality calculations for Pyramids
 *
 * This file is part of VERDICT
 *
 */

#define VERDICT_EXPORTS

#include "moab/verdict.h"
#include "VerdictVector.hpp"
#include <memory.h>

/*
  the pyramid element

       5
       ^
       |\
      /| \\_
     |  \   \
     |  | \_ \_
     /   \/4\  \
    |   /|    \ \_
    | /  \      \ \
    /     |       \
  1 \_    |      _/3
      \_   \   _/
        \_ | _/
          \_/
          2

    a quadrilateral base and a pointy peak like a pyramid

*/

/*!
  the volume of a pyramid

  the volume is calculated by dividing the pyramid into
  2 tets and summing the volumes of the 2 tets.
*/

C_FUNC_DEF double v_pyramid_volume( int num_nodes, double coordinates[][3] )
{

    double volume = 0;
    VerdictVector side1, side2, side3;

    if( num_nodes == 5 )
    {
        // divide the pyramid into 2 tets and calculate each

        side1.set( coordinates[1][0] - coordinates[0][0], coordinates[1][1] - coordinates[0][1],
                   coordinates[1][2] - coordinates[0][2] );

        side2.set( coordinates[3][0] - coordinates[0][0], coordinates[3][1] - coordinates[0][1],
                   coordinates[3][2] - coordinates[0][2] );

        side3.set( coordinates[4][0] - coordinates[0][0], coordinates[4][1] - coordinates[0][1],
                   coordinates[4][2] - coordinates[0][2] );

        // volume of the first tet
        volume = ( side3 % ( side1 * side2 ) ) / 6.0;

        side1.set( coordinates[3][0] - coordinates[2][0], coordinates[3][1] - coordinates[2][1],
                   coordinates[3][2] - coordinates[2][2] );

        side2.set( coordinates[1][0] - coordinates[2][0], coordinates[1][1] - coordinates[2][1],
                   coordinates[1][2] - coordinates[2][2] );

        side3.set( coordinates[4][0] - coordinates[2][0], coordinates[4][1] - coordinates[2][1],
                   coordinates[4][2] - coordinates[2][2] );

        // volume of the second tet
        volume += ( side3 % ( side1 * side2 ) ) / 6.0;
    }
    return (double)volume;
}

C_FUNC_DEF void v_pyramid_quality( int num_nodes, double coordinates[][3], unsigned int metrics_request_flag,
                                   PyramidMetricVals* metric_vals )
{
    memset( metric_vals, 0, sizeof( PyramidMetricVals ) );

    if( metrics_request_flag & V_PYRAMID_VOLUME ) metric_vals->volume = v_pyramid_volume( num_nodes, coordinates );
}
