//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
//
// $Id: G4Orb.hh 105148 2017-07-14 08:35:13Z gcosmo $
//
//
// --------------------------------------------------------------------
// GEANT 4 class header file
//
// G4Orb
//
// Class description:
//
//   A G4Orb is a simple case of G4Sphere. It has only:
//   fRmax  outer radius

//  History:
// 20.08.03 V.Grichine - created
// --------------------------------------------------------------------

#ifndef G4Orb_HH
#define G4Orb_HH

#if defined(G4GEOM_USE_USOLIDS)
#define G4GEOM_USE_UORB 1
#endif

#if defined(G4GEOM_USE_UORB)
  #define G4UOrb G4Orb
  #include "G4UOrb.hh"
#else

#include <CLHEP/Units/PhysicalConstants.h>

#include "G4CSGSolid.hh"
#include "G4Polyhedron.hh"

class G4Orb : public G4CSGSolid
{
  public:  // with description

    G4Orb(const G4String& pName, G4double pRmax);
       
    virtual ~G4Orb() ;
    
    // Accessors
       
    inline G4double GetRadius() const;
    inline G4double GetRadialTolerance() const;

    // Modifiers

    inline void SetRadius(G4double newRmax);

    // Methods for solid

    inline G4double GetCubicVolume();
    inline G4double GetSurfaceArea();

    void ComputeDimensions(      G4VPVParameterisation* p,
                           const G4int n,
                           const G4VPhysicalVolume* pRep);

    void Extent(G4ThreeVector& pMin, G4ThreeVector& pMax) const;

    G4bool CalculateExtent(const EAxis pAxis,
                           const G4VoxelLimits& pVoxelLimit,
                           const G4AffineTransform& pTransform,
                                 G4double& pmin, G4double& pmax) const;
         
    EInside Inside(const G4ThreeVector& p) const;

    G4ThreeVector SurfaceNormal( const G4ThreeVector& p) const;

    G4double DistanceToIn(const G4ThreeVector& p,
                          const G4ThreeVector& v) const;
    
    G4double DistanceToIn(const G4ThreeVector& p) const;
    
    G4double DistanceToOut(const G4ThreeVector& p,
                           const G4ThreeVector& v,
                           const G4bool calcNorm=G4bool(false),
                                 G4bool *validNorm=0,
                                 G4ThreeVector *n=0) const;
         
    G4double DistanceToOut(const G4ThreeVector& p) const;

    G4GeometryType GetEntityType() const;
 
    G4ThreeVector GetPointOnSurface() const;

    G4VSolid* Clone() const;

    std::ostream& StreamInfo(std::ostream& os) const;

    // Visualisation functions
  
    void          DescribeYourselfTo(G4VGraphicsScene& scene) const;
    G4Polyhedron* CreatePolyhedron() const;

  public:  // without description

    G4Orb(__void__&);
      // Fake default constructor for usage restricted to direct object
      // persistency for clients requiring preallocation of memory for
      // persistifiable objects.

    G4Orb(const G4Orb& rhs);
    G4Orb& operator=(const G4Orb& rhs); 
      // Copy constructor and assignment operator.

  protected:
  
    // Used by distanceToOut
  
    enum ESide {kNull,kRMax};
  
    // used by normal
  
    enum ENorm {kNRMax};

  private:

    G4double fRmax;
    G4double fRmaxTolerance;
};

#include "G4Orb.icc"

#endif

#endif
