/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { describe, it, expect } from 'vitest';
import { toolsCommand } from './toolsCommand.js';
import { createMockCommandContext } from '../../test-utils/mockCommandContext.js';
import { MessageType } from '../types.js';
// Mock tools for testing
const mockTools = [
    {
        name: 'file-reader',
        displayName: 'File Reader',
        description: 'Reads files from the local system.',
        schema: {},
    },
    {
        name: 'code-editor',
        displayName: 'Code Editor',
        description: 'Edits code files.',
        schema: {},
    },
];
describe('toolsCommand', () => {
    it('should display an error if the tool registry is unavailable', async () => {
        const mockContext = createMockCommandContext({
            services: {
                config: {
                    getToolRegistry: () => undefined,
                },
            },
        });
        if (!toolsCommand.action)
            throw new Error('Action not defined');
        await toolsCommand.action(mockContext, '');
        expect(mockContext.ui.addItem).toHaveBeenCalledWith({
            type: MessageType.ERROR,
            text: 'Could not retrieve tool registry.',
        });
    });
    it('should display "No tools available" when none are found', async () => {
        const mockContext = createMockCommandContext({
            services: {
                config: {
                    getToolRegistry: () => ({
                        getAllTools: () => [],
                    }),
                },
            },
        });
        if (!toolsCommand.action)
            throw new Error('Action not defined');
        await toolsCommand.action(mockContext, '');
        expect(mockContext.ui.addItem).toHaveBeenCalledWith({
            type: MessageType.TOOLS_LIST,
            tools: [],
            showDescriptions: false,
        });
    });
    it('should list tools without descriptions by default', async () => {
        const mockContext = createMockCommandContext({
            services: {
                config: {
                    getToolRegistry: () => ({ getAllTools: () => mockTools }),
                },
            },
        });
        if (!toolsCommand.action)
            throw new Error('Action not defined');
        await toolsCommand.action(mockContext, '');
        const [message] = mockContext.ui.addItem.mock
            .calls[0];
        expect(message.type).toBe(MessageType.TOOLS_LIST);
        expect(message.showDescriptions).toBe(false);
        expect(message.tools).toHaveLength(2);
        expect(message.tools[0].displayName).toBe('File Reader');
        expect(message.tools[1].displayName).toBe('Code Editor');
    });
    it('should list tools with descriptions when "desc" arg is passed', async () => {
        const mockContext = createMockCommandContext({
            services: {
                config: {
                    getToolRegistry: () => ({ getAllTools: () => mockTools }),
                },
            },
        });
        if (!toolsCommand.action)
            throw new Error('Action not defined');
        await toolsCommand.action(mockContext, 'desc');
        const [message] = mockContext.ui.addItem.mock
            .calls[0];
        expect(message.type).toBe(MessageType.TOOLS_LIST);
        expect(message.showDescriptions).toBe(true);
        expect(message.tools).toHaveLength(2);
        expect(message.tools[0].description).toBe('Reads files from the local system.');
        expect(message.tools[1].description).toBe('Edits code files.');
    });
});
//# sourceMappingURL=toolsCommand.test.js.map