# 15 "src/ocaml/preprocess/lexer_raw.mll"
 
open Misc
open Std
open Lexing
open Parser_raw

type keywords = (string, Parser_raw.token) Hashtbl.t

type error =
  | Illegal_character of char
  | Illegal_escape of string * string option
  | Reserved_sequence of string * string option
  | Unterminated_comment of Location.t
  | Unterminated_string
  | Unterminated_string_in_comment of Location.t * Location.t
  | Empty_character_literal
  | Keyword_as_label of string
  | Invalid_literal of string

exception Error of error * Location.t

(* Monad in which the lexer evaluates *)
type 'a result =
  | Return of 'a
  | Refill of (unit -> 'a result)
  | Fail of error * Location.t

let return a = Return a

let fail lexbuf e = Fail (e, Location.curr lexbuf)
let fail_loc e l = Fail (e,l)

let rec (>>=) (m : 'a result) (f : 'a -> 'b result) : 'b result =
  match m with
  | Return a -> f a
  | Refill u ->
    Refill (fun () -> u () >>= f)
  | Fail _ as e -> e

type preprocessor = (Lexing.lexbuf -> Parser_raw.token) -> Lexing.lexbuf -> Parser_raw.token

type state = {
  keywords: keywords;
  mutable buffer: Buffer.t;
  mutable string_start_loc: Location.t;
  mutable comment_start_loc: Location.t list;
  mutable preprocessor: preprocessor option;
}

let make ?preprocessor keywords = {
  keywords;
  buffer = Buffer.create 17;
  string_start_loc = Location.none;
  comment_start_loc = [];
  preprocessor;
}

let lABEL m = m >>= fun v -> return (LABEL v)
let oPTLABEL m = m >>= fun v -> return (OPTLABEL v)

let rec catch m f = match m with
  | Fail (e,l) -> f e l
  | Refill next -> Refill (fun () -> catch (next ()) f)
  | Return _ -> m

(* The table of keywords *)

let keyword_table : keywords =
  create_hashtable 149 [
    "and", AND;
    "as", AS;
    "assert", ASSERT;
    "begin", BEGIN;
    "class", CLASS;
    "constraint", CONSTRAINT;
    "do", DO;
    "done", DONE;
    "downto", DOWNTO;
    "else", ELSE;
    "end", END;
    "exception", EXCEPTION;
    "external", EXTERNAL;
    "false", FALSE;
    "for", FOR;
    "fun", FUN;
    "function", FUNCTION;
    "functor", FUNCTOR;
    "if", IF;
    "in", IN;
    "include", INCLUDE;
    "inherit", INHERIT;
    "initializer", INITIALIZER;
    "lazy", LAZY;
    "let", LET;
    "match", MATCH;
    "method", METHOD;
    "module", MODULE;
    "mutable", MUTABLE;
    "new", NEW;
    "nonrec", NONREC;
    "object", OBJECT;
    "of", OF;
    "open", OPEN;
    "or", OR;
(*  "parser", PARSER; *)
    "private", PRIVATE;
    "rec", REC;
    "sig", SIG;
    "struct", STRUCT;
    "then", THEN;
    "to", TO;
    "true", TRUE;
    "try", TRY;
    "type", TYPE;
    "val", VAL;
    "virtual", VIRTUAL;
    "when", WHEN;
    "while", WHILE;
    "with", WITH;

    "lor", INFIXOP3("lor"); (* Should be INFIXOP2 *)
    "lxor", INFIXOP3("lxor"); (* Should be INFIXOP2 *)
    "mod", INFIXOP3("mod");
    "land", INFIXOP3("land");
    "lsl", INFIXOP4("lsl");
    "lsr", INFIXOP4("lsr");
    "asr", INFIXOP4("asr");
]

let keywords l = create_hashtable 11 l

let list_keywords =
  let add_kw str _tok kws = str :: kws in
  let init = Hashtbl.fold add_kw keyword_table [] in
  fun keywords ->
    Hashtbl.fold add_kw keywords init

(* To store the position of the beginning of a string and comment *)
let in_comment state = state.comment_start_loc <> []

(* Escaped chars are interpreted in strings unless they are in comments. *)
let store_escaped_uchar state lexbuf u =
  if in_comment state
  then Buffer.add_string state.buffer (Lexing.lexeme lexbuf)
  else Buffer.add_utf_8_uchar state.buffer u


let compute_quoted_string_idloc {Location.loc_start = orig_loc; _ } shift id =
  let id_start_pos = orig_loc.Lexing.pos_cnum + shift in
  let loc_start =
    Lexing.{orig_loc with pos_cnum = id_start_pos }
  in
  let loc_end =
    Lexing.{orig_loc with pos_cnum = id_start_pos + String.length id }
  in
  {Location. loc_start ; loc_end ; loc_ghost = false }

let wrap_string_lexer f state lexbuf =
  Buffer.reset state.buffer;
  state.string_start_loc <- Location.curr lexbuf;
  f state lexbuf >>= fun loc_end ->
  lexbuf.lex_start_p <- state.string_start_loc.Location.loc_start;
  let loc =
    Location.{
      loc_ghost = false;
      loc_start = state.string_start_loc.Location.loc_end;
      loc_end;
    }
  in
  state.string_start_loc <- Location.none;
  return (Buffer.contents state.buffer, loc)

(* to translate escape sequences *)

let digit_value c =
  match c with
  | 'a' .. 'f' -> 10 + Char.code c - Char.code 'a'
  | 'A' .. 'F' -> 10 + Char.code c - Char.code 'A'
  | '0' .. '9' -> Char.code c - Char.code '0'
  | _ -> assert false

let num_value lexbuf ~base ~first ~last =
  let c = ref 0 in
  for i = first to last do
    let v = digit_value (Lexing.lexeme_char lexbuf i) in
    assert(v < base);
    c := (base * !c) + v
  done;
  !c

let char_for_backslash = function
  | 'n' -> '\010'
  | 'r' -> '\013'
  | 'b' -> '\008'
  | 't' -> '\009'
  | c   -> c

let illegal_escape lexbuf reason =
  let error = Illegal_escape (Lexing.lexeme lexbuf, Some reason) in
  fail lexbuf error

let char_for_decimal_code state lexbuf i =
  let c = num_value lexbuf ~base:10 ~first:i ~last:(i+2) in
  if (c < 0 || c > 255) then
    if in_comment state
    then return 'x'
    else
      illegal_escape lexbuf
        (Printf.sprintf
          "%d is outside the range of legal characters (0-255)." c)
  else return (Char.chr c)

let char_for_octal_code state lexbuf i =
  let c = num_value lexbuf ~base:8 ~first:i ~last:(i+2) in
  if (c < 0 || c > 255) then
    if in_comment state
    then return 'x'
    else
      illegal_escape lexbuf
        (Printf.sprintf
          "o%o (=%d) is outside the range of legal characters (0-255)." c c)
  else return (Char.chr c)

let char_for_hexadecimal_code lexbuf i =
  Char.chr (num_value lexbuf ~base:16 ~first:i ~last:(i+1))

let uchar_for_uchar_escape lexbuf =
  let illegal_escape lexbuf reason =
    let error = Illegal_escape (Lexing.lexeme lexbuf, Some reason) in
    raise (Error (error, Location.curr lexbuf))
  in
  let len = Lexing.lexeme_end lexbuf - Lexing.lexeme_start lexbuf in
  let first = 3 (* skip opening \u{ *) in
  let last = len - 2 (* skip closing } *) in
  let digit_count = last - first + 1 in
  match digit_count > 6 with
  | true ->
      illegal_escape lexbuf
        "too many digits, expected 1 to 6 hexadecimal digits"
  | false ->
      let cp = num_value lexbuf ~base:16 ~first ~last in
      if Uchar.is_valid cp then Uchar.unsafe_of_int cp else
      illegal_escape lexbuf
        (Printf.sprintf "%X is not a Unicode scalar value" cp)

let keyword_or state s default =
  try Hashtbl.find state.keywords s
      with Not_found -> try Hashtbl.find keyword_table s
  with Not_found -> default

let is_keyword name = Hashtbl.mem keyword_table name

let check_label_name lexbuf name =
  if is_keyword name
  then fail lexbuf (Keyword_as_label name)
  else return name

(* Update the current location with file name and line number. *)

let update_loc lexbuf _file line absolute chars =
  let pos = lexbuf.lex_curr_p in
  let new_file = pos.pos_fname
    (*match file with
      | None -> pos.pos_fname
      | Some s -> s*)
  in
  lexbuf.lex_curr_p <- { pos with
    pos_fname = new_file;
    pos_lnum = if absolute then line else pos.pos_lnum + line;
    pos_bol = pos.pos_cnum - chars;
  }
;;

(* Warn about Latin-1 characters used in idents *)

let warn_latin1 lexbuf =
  Location.deprecated (Location.curr lexbuf)
    "ISO-Latin1 characters in identifiers"
;;

(* Error report *)

open Format

let prepare_error loc = function
  | Illegal_character c ->
      Location.errorf ~loc "Illegal character (%s)" (Char.escaped c)
  | Illegal_escape (s, explanation) ->
      Location.errorf ~loc
        "Illegal backslash escape in string or character (%s)%t" s
        (fun ppf -> match explanation with
           | None -> ()
           | Some expl -> fprintf ppf ": %s" expl)
  | Reserved_sequence (s, explanation) ->
      Location.errorf ~loc
        "Reserved character sequence: %s%t" s
        (fun ppf -> match explanation with
           | None -> ()
           | Some expl -> fprintf ppf " %s" expl)
  | Unterminated_comment _ ->
      Location.errorf ~loc "Comment not terminated"
  | Unterminated_string ->
      Location.errorf ~loc "String literal not terminated"
  | Unterminated_string_in_comment (_, literal_loc) ->
      Location.errorf ~loc
        "This comment contains an unterminated string literal"
        ~sub:[Location.msg ~loc:literal_loc "String literal begins here"]
  | Empty_character_literal ->
      let msg = "Illegal empty character literal ''" in
      let sub =
        [Location.msg
           "Hint: Did you mean ' ' or a type variable 'a?"] in
      Location.error ~loc ~sub msg
  | Keyword_as_label kwd ->
      Location.errorf ~loc
        "`%s' is a keyword, it cannot be used as label name" kwd
  | Invalid_literal s ->
      Location.errorf ~loc "Invalid literal %s" s
(* FIXME: Invalid_directive? *)

let () =
  Location.register_error_of_exn
    (function
      | Error (err, loc) ->
        Some (prepare_error loc err)
      | _ ->
        None
    )


# 333 "src/ocaml/preprocess/lexer_raw.ml"
let __ocaml_lex_refill : (Lexing.lexbuf -> 'a) -> (Lexing.lexbuf -> 'a) =
# 393 "src/ocaml/preprocess/lexer_raw.mll"
       (fun k lexbuf -> Refill (fun () -> k lexbuf))
# 337 "src/ocaml/preprocess/lexer_raw.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\155\255\156\255\224\000\003\001\038\001\073\001\108\001\
    \143\001\181\255\178\001\189\255\091\000\215\001\250\001\068\000\
    \071\000\029\002\207\255\209\255\212\255\064\002\159\002\194\002\
    \088\000\224\002\255\000\233\255\093\003\221\003\099\004\227\004\
    \105\005\233\005\111\006\239\006\025\007\248\007\248\008\117\007\
    \084\008\122\000\254\255\001\000\005\000\255\255\006\000\007\000\
    \203\255\029\009\250\255\252\255\180\255\182\255\064\009\099\009\
    \249\009\121\010\247\255\246\255\255\010\127\011\244\255\243\255\
    \005\012\133\012\109\002\011\013\139\013\176\013\049\014\177\014\
    \075\015\203\015\081\016\209\016\087\017\215\017\176\018\048\019\
    \202\019\087\000\074\020\208\020\080\021\234\021\106\022\108\000\
    \203\022\106\000\107\000\011\000\232\255\226\255\225\255\220\255\
    \099\004\118\000\099\000\113\000\224\255\128\000\136\000\208\000\
    \223\255\224\000\219\000\222\255\227\004\220\000\221\255\187\255\
    \051\018\231\255\192\002\233\005\108\023\229\022\230\255\084\008\
    \236\023\229\255\097\009\114\024\153\024\228\255\249\009\024\025\
    \227\255\174\013\222\000\218\255\061\025\217\255\096\025\161\005\
    \127\002\216\255\012\000\024\001\034\001\045\001\043\001\216\255\
    \013\000\131\025\166\025\201\025\236\025\199\255\200\255\201\255\
    \197\255\015\026\227\000\219\000\190\255\191\255\192\255\225\000\
    \177\255\175\255\184\255\050\026\085\026\120\026\155\026\190\026\
    \252\026\241\255\120\027\243\255\017\000\244\255\047\001\167\022\
    \253\255\250\000\000\001\255\255\254\255\252\255\156\027\242\027\
    \121\028\121\010\160\028\127\011\031\029\049\029\004\001\012\001\
    \018\000\251\255\250\255\249\255\128\006\039\001\194\002\013\001\
    \248\255\214\002\020\001\247\255\224\002\122\004\030\001\246\255\
    \117\007\031\001\245\255\094\001\104\001\246\255\247\255\248\255\
    \095\001\132\029\255\255\249\255\205\000\255\010\130\004\253\255\
    \094\001\111\001\112\001\154\004\252\255\005\012\251\255\133\012\
    \194\014\250\255\130\001\139\001\252\255\219\007\254\255\255\255\
    \146\001\147\001\253\255\095\029\038\001\045\001\103\001\113\001\
    \049\001\132\001\058\001\019\000\255\255";
  Lexing.lex_backtrk =
   "\255\255\255\255\255\255\095\000\094\000\091\000\090\000\083\000\
    \081\000\255\255\069\000\255\255\062\000\061\000\059\000\057\000\
    \053\000\086\000\255\255\255\255\255\255\041\000\040\000\047\000\
    \045\000\067\000\044\000\255\255\017\000\017\000\016\000\015\000\
    \014\000\013\000\013\000\013\000\010\000\007\000\006\000\072\000\
    \050\000\002\000\255\255\100\000\100\000\255\255\255\255\255\255\
    \255\255\051\000\255\255\255\255\255\255\255\255\004\000\089\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \013\000\013\000\097\000\013\000\013\000\098\000\021\000\021\000\
    \019\000\018\000\021\000\018\000\018\000\017\000\019\000\018\000\
    \019\000\255\255\020\000\020\000\017\000\017\000\019\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \035\000\035\000\035\000\035\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\036\000\255\255\093\000\255\255\096\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\042\000\092\000\087\000\049\000\255\255\255\255\255\255\
    \255\255\060\000\079\000\076\000\255\255\255\255\255\255\077\000\
    \255\255\255\255\255\255\070\000\088\000\082\000\085\000\084\000\
    \255\255\255\255\013\000\255\255\014\000\255\255\014\000\014\000\
    \255\255\014\000\014\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \009\000\009\000\255\255\255\255\006\000\006\000\006\000\255\255\
    \001\000\006\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\003\000\255\255\255\255\
    \003\000\255\255\255\255\255\255\002\000\255\255\255\255\001\000\
    \255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_default =
   "\001\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\089\000\000\000\070\000\070\000\030\000\031\000\
    \032\000\033\000\033\000\033\000\255\255\255\255\033\000\255\255\
    \255\255\255\255\000\000\255\255\255\255\000\000\255\255\255\255\
    \000\000\255\255\000\000\000\000\000\000\000\000\255\255\255\255\
    \056\000\057\000\000\000\000\000\060\000\061\000\000\000\000\000\
    \033\000\033\000\255\255\033\000\033\000\255\255\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\255\255\070\000\070\000\070\000\070\000\070\000\255\255\
    \095\000\255\255\255\255\255\255\000\000\000\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \000\000\255\255\255\255\000\000\255\255\255\255\000\000\000\000\
    \255\255\000\000\255\255\255\255\116\000\255\255\000\000\255\255\
    \120\000\000\000\255\255\123\000\255\255\000\000\255\255\127\000\
    \000\000\255\255\255\255\000\000\255\255\000\000\255\255\139\000\
    \255\255\000\000\255\255\139\000\140\000\139\000\142\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\000\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \169\000\000\000\170\000\000\000\255\255\000\000\190\000\255\255\
    \000\000\255\255\255\255\000\000\000\000\000\000\255\255\255\255\
    \184\000\255\255\255\255\255\255\188\000\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\000\000\255\255\255\255\255\255\000\000\
    \255\255\255\255\000\000\255\255\213\000\000\000\000\000\000\000\
    \255\255\219\000\000\000\000\000\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\000\000\255\255\000\000\255\255\
    \255\255\000\000\255\255\236\000\000\000\255\255\000\000\000\000\
    \255\255\255\255\000\000\255\255\255\255\255\255\246\000\249\000\
    \255\255\249\000\255\255\255\255\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\041\000\042\000\042\000\041\000\043\000\047\000\045\000\
    \045\000\042\000\046\000\046\000\047\000\090\000\137\000\143\000\
    \091\000\138\000\144\000\171\000\191\000\252\000\211\000\192\000\
    \041\000\008\000\027\000\022\000\006\000\004\000\021\000\026\000\
    \024\000\019\000\023\000\007\000\018\000\017\000\040\000\003\000\
    \029\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\016\000\015\000\014\000\013\000\039\000\036\000\
    \005\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\012\000\044\000\011\000\005\000\038\000\
    \020\000\034\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\035\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\025\000\010\000\009\000\037\000\152\000\
    \154\000\151\000\130\000\041\000\150\000\149\000\041\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\094\000\093\000\101\000\101\000\101\000\101\000\157\000\
    \100\000\156\000\041\000\155\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\105\000\105\000\
    \105\000\105\000\105\000\105\000\105\000\105\000\105\000\105\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \103\000\103\000\103\000\103\000\103\000\103\000\103\000\103\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\158\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\104\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \002\000\003\000\107\000\110\000\003\000\003\000\003\000\131\000\
    \161\000\090\000\003\000\003\000\091\000\003\000\003\000\003\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\003\000\159\000\003\000\003\000\003\000\003\000\
    \003\000\160\000\137\000\180\000\003\000\138\000\092\000\003\000\
    \003\000\003\000\179\000\195\000\137\000\003\000\003\000\138\000\
    \003\000\003\000\003\000\194\000\200\000\143\000\141\000\137\000\
    \144\000\191\000\138\000\203\000\192\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\142\000\207\000\210\000\005\000\
    \231\000\245\000\005\000\005\000\005\000\141\000\246\000\140\000\
    \005\000\005\000\250\000\005\000\005\000\005\000\193\000\204\000\
    \204\000\204\000\204\000\088\000\003\000\251\000\003\000\000\000\
    \005\000\003\000\005\000\005\000\005\000\005\000\005\000\224\000\
    \171\000\215\000\006\000\211\000\234\000\006\000\006\000\006\000\
    \000\000\247\000\215\000\006\000\006\000\216\000\006\000\006\000\
    \006\000\224\000\224\000\248\000\226\000\226\000\224\000\003\000\
    \000\000\003\000\000\000\006\000\005\000\006\000\006\000\006\000\
    \006\000\006\000\218\000\189\000\215\000\146\000\248\000\234\000\
    \146\000\146\000\146\000\000\000\000\000\239\000\146\000\146\000\
    \240\000\146\000\167\000\146\000\239\000\239\000\000\000\241\000\
    \241\000\000\000\005\000\000\000\005\000\000\000\146\000\006\000\
    \146\000\166\000\146\000\146\000\146\000\000\000\000\000\000\000\
    \164\000\000\000\164\000\164\000\164\000\164\000\000\000\000\000\
    \000\000\164\000\164\000\000\000\164\000\164\000\164\000\000\000\
    \000\000\000\000\000\000\000\000\217\000\006\000\000\000\006\000\
    \000\000\164\000\146\000\164\000\165\000\164\000\164\000\164\000\
    \000\000\000\000\000\000\006\000\000\000\000\000\006\000\006\000\
    \006\000\000\000\000\000\000\000\006\000\006\000\000\000\006\000\
    \006\000\006\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \146\000\000\000\146\000\000\000\006\000\164\000\006\000\006\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\000\000\000\000\
    \006\000\000\000\000\000\006\000\006\000\006\000\000\000\255\255\
    \000\000\006\000\006\000\000\000\006\000\006\000\006\000\237\000\
    \000\000\000\000\000\000\164\000\000\000\164\000\000\000\162\000\
    \006\000\006\000\000\000\006\000\006\000\006\000\006\000\006\000\
    \255\255\000\000\000\000\006\000\000\000\000\000\006\000\006\000\
    \006\000\000\000\255\255\000\000\006\000\006\000\000\000\153\000\
    \006\000\006\000\000\000\255\255\000\000\255\255\163\000\255\255\
    \006\000\000\000\000\000\000\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\146\000\000\000\
    \000\000\146\000\146\000\146\000\000\000\000\000\000\000\146\000\
    \146\000\000\000\146\000\147\000\146\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\006\000\000\000\006\000\000\000\146\000\
    \006\000\146\000\146\000\148\000\146\000\146\000\000\000\000\000\
    \000\000\006\000\000\000\000\000\006\000\006\000\145\000\255\255\
    \214\000\000\000\006\000\006\000\000\000\006\000\006\000\006\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\006\000\000\000\
    \006\000\000\000\006\000\146\000\006\000\006\000\006\000\006\000\
    \006\000\000\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \136\000\000\000\000\000\238\000\000\000\000\000\066\000\000\000\
    \000\000\066\000\066\000\066\000\000\000\000\000\000\000\066\000\
    \066\000\146\000\066\000\146\000\066\000\000\000\006\000\136\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\066\000\
    \136\000\000\000\066\000\066\000\066\000\066\000\000\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\000\000\000\000\000\000\006\000\000\000\006\000\136\000\
    \134\000\000\000\134\000\134\000\134\000\134\000\000\000\000\000\
    \000\000\134\000\134\000\066\000\134\000\134\000\134\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\134\000\000\000\134\000\134\000\134\000\134\000\134\000\
    \000\000\000\000\000\000\003\000\000\000\000\000\003\000\003\000\
    \003\000\066\000\000\000\133\000\132\000\003\000\000\000\003\000\
    \003\000\003\000\201\000\201\000\201\000\201\000\201\000\201\000\
    \201\000\201\000\201\000\201\000\003\000\134\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\112\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \000\000\000\000\000\000\134\000\111\000\134\000\000\000\114\000\
    \003\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\000\000\113\000\000\000\003\000\114\000\
    \003\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\000\000\113\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\072\000\255\255\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\000\000\
    \000\000\000\000\071\000\000\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \255\255\255\255\255\255\255\255\028\000\255\255\000\000\000\000\
    \000\000\000\000\071\000\000\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\072\000\255\255\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\074\000\
    \000\000\000\000\071\000\000\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\075\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\076\000\073\000\073\000\
    \255\255\255\255\255\255\255\255\028\000\255\255\000\000\074\000\
    \000\000\000\000\071\000\000\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\075\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\076\000\073\000\073\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\108\000\108\000\108\000\108\000\
    \108\000\108\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\227\000\227\000\227\000\227\000\227\000\227\000\
    \227\000\227\000\227\000\227\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\000\000\255\255\108\000\108\000\108\000\108\000\
    \108\000\108\000\228\000\228\000\228\000\228\000\228\000\228\000\
    \228\000\228\000\228\000\228\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\109\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\109\000\109\000\109\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\109\000\109\000\109\000\109\000\
    \109\000\109\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\000\000\255\255\109\000\109\000\109\000\109\000\
    \109\000\109\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\141\000\137\000\000\000\000\000\138\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\141\000\000\000\140\000\255\255\255\255\255\255\255\255\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\255\255\255\255\255\255\255\255\
    \123\000\255\255\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \208\000\208\000\208\000\208\000\208\000\208\000\208\000\208\000\
    \208\000\208\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\208\000\208\000\208\000\208\000\208\000\208\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\067\000\000\000\000\000\
    \000\000\208\000\208\000\208\000\208\000\208\000\208\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\055\000\000\000\055\000\055\000\055\000\055\000\
    \000\000\000\000\000\000\055\000\055\000\000\000\055\000\055\000\
    \055\000\000\000\255\255\255\255\255\255\255\255\000\000\255\255\
    \000\000\000\000\000\000\055\000\064\000\055\000\055\000\055\000\
    \055\000\055\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\055\000\
    \061\000\000\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\000\000\055\000\006\000\055\000\
    \000\000\006\000\006\000\006\000\000\000\000\000\000\000\006\000\
    \006\000\000\000\006\000\054\000\006\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\006\000\
    \000\000\006\000\006\000\006\000\006\000\006\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\053\000\006\000\000\000\000\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \000\000\006\000\052\000\006\000\000\000\000\000\000\000\000\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \000\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\055\000\000\000\055\000\055\000\055\000\055\000\000\000\
    \000\000\000\000\055\000\055\000\000\000\055\000\055\000\055\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\055\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\000\000\243\000\000\000\243\000\243\000\243\000\243\000\
    \243\000\243\000\243\000\243\000\243\000\243\000\243\000\243\000\
    \243\000\243\000\243\000\243\000\243\000\243\000\243\000\243\000\
    \243\000\243\000\243\000\243\000\243\000\243\000\055\000\057\000\
    \242\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\000\000\055\000\049\000\055\000\000\000\
    \049\000\049\000\049\000\000\000\000\000\000\000\049\000\049\000\
    \000\000\049\000\048\000\049\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\049\000\000\000\
    \051\000\049\000\049\000\049\000\049\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\000\000\
    \000\000\000\000\049\000\120\000\000\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\000\000\
    \049\000\000\000\050\000\000\000\000\000\000\000\000\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\000\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\049\000\000\000\
    \000\000\049\000\049\000\049\000\000\000\000\000\000\000\049\000\
    \049\000\000\000\049\000\049\000\049\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\049\000\
    \255\255\049\000\049\000\049\000\049\000\049\000\000\000\000\000\
    \000\000\006\000\000\000\000\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\006\000\006\000\000\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\006\000\049\000\006\000\006\000\006\000\006\000\
    \006\000\000\000\000\000\000\000\055\000\000\000\055\000\055\000\
    \055\000\055\000\000\000\000\000\000\000\055\000\055\000\000\000\
    \055\000\055\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\049\000\000\000\049\000\000\000\055\000\006\000\055\000\
    \055\000\055\000\055\000\055\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\006\000\000\000\006\000\000\000\
    \122\000\055\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\000\000\121\000\000\000\055\000\
    \000\000\055\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\059\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\255\255\255\255\255\255\255\255\
    \127\000\255\255\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\058\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\255\255\255\255\255\255\255\255\
    \184\000\255\255\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\229\000\
    \229\000\229\000\229\000\229\000\229\000\229\000\229\000\229\000\
    \229\000\063\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \229\000\229\000\229\000\229\000\229\000\229\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\000\000\255\255\
    \229\000\229\000\229\000\229\000\229\000\229\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\062\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \188\000\188\000\188\000\188\000\188\000\188\000\188\000\188\000\
    \188\000\188\000\188\000\188\000\188\000\188\000\188\000\188\000\
    \188\000\188\000\188\000\188\000\188\000\188\000\188\000\188\000\
    \188\000\188\000\255\255\255\255\255\255\255\255\188\000\255\255\
    \188\000\188\000\188\000\188\000\188\000\188\000\188\000\188\000\
    \188\000\188\000\188\000\188\000\188\000\188\000\188\000\188\000\
    \188\000\188\000\188\000\188\000\188\000\188\000\188\000\188\000\
    \188\000\188\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\230\000\230\000\230\000\
    \230\000\230\000\230\000\230\000\230\000\230\000\230\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\230\000\230\000\
    \230\000\230\000\230\000\230\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\000\000\255\255\230\000\230\000\
    \230\000\230\000\230\000\230\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\065\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\066\000\255\255\066\000\000\000\255\255\255\255\066\000\
    \066\000\255\255\066\000\255\255\066\000\232\000\232\000\232\000\
    \232\000\232\000\232\000\232\000\232\000\232\000\232\000\255\255\
    \255\255\066\000\066\000\066\000\255\255\066\000\232\000\232\000\
    \232\000\232\000\232\000\232\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\255\255\255\255\066\000\000\000\255\255\232\000\232\000\
    \232\000\232\000\232\000\232\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\066\000\255\255\255\255\255\255\255\255\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\000\000\255\255\000\000\000\000\000\000\068\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\069\000\
    \255\255\069\000\000\000\255\255\255\255\069\000\069\000\255\255\
    \069\000\255\255\069\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\069\000\
    \069\000\069\000\255\255\069\000\000\000\000\000\000\000\000\000\
    \000\000\069\000\000\000\000\000\069\000\069\000\069\000\000\000\
    \000\000\000\000\069\000\069\000\000\000\069\000\000\000\069\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\069\000\069\000\255\255\000\000\069\000\069\000\069\000\
    \069\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\069\000\
    \255\255\255\255\255\255\255\255\000\000\129\000\069\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\000\000\128\000\000\000\069\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\087\000\255\255\087\000\255\255\
    \255\255\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\232\000\232\000\232\000\232\000\232\000\232\000\
    \232\000\232\000\232\000\232\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\232\000\232\000\232\000\232\000\232\000\
    \232\000\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\232\000\232\000\232\000\232\000\232\000\
    \232\000\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\233\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\000\000\000\000\
    \071\000\000\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\255\255\255\255\
    \255\255\255\255\072\000\255\255\000\000\000\000\000\000\000\000\
    \071\000\255\255\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\000\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\085\000\085\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \077\000\077\000\077\000\077\000\077\000\077\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\000\000\255\255\
    \077\000\077\000\077\000\077\000\077\000\077\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\078\000\255\255\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \077\000\077\000\077\000\077\000\077\000\077\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\079\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\255\255\255\255\255\255\255\255\077\000\255\255\
    \077\000\077\000\077\000\077\000\077\000\077\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\079\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \115\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\000\000\000\000\
    \000\000\000\000\116\000\000\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\078\000\078\000\078\000\078\000\078\000\078\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \083\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\255\255\255\255\255\255\255\255\078\000\
    \255\255\078\000\078\000\078\000\078\000\078\000\078\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \083\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\081\000\255\255\081\000\255\255\255\255\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\000\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\255\255\255\255\255\255\
    \255\255\080\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \255\255\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\255\255\
    \255\255\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\000\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\081\000\255\255\081\000\255\255\255\255\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\000\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\255\255\255\255\255\255\255\255\084\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\085\000\085\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\255\255\255\255\255\255\
    \255\255\085\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \255\255\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\255\255\255\255\255\255\
    \255\255\086\000\255\255\000\000\183\000\000\000\000\000\000\000\
    \000\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\099\000\000\000\099\000\117\000\000\000\
    \000\000\117\000\099\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\097\000\097\000\097\000\097\000\097\000\
    \097\000\097\000\097\000\097\000\097\000\117\000\182\000\000\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\000\000\181\000\000\000\000\000\000\000\099\000\
    \000\000\000\000\000\000\000\000\000\000\099\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\099\000\098\000\000\000\000\000\099\000\000\000\099\000\
    \000\000\000\000\000\000\096\000\122\000\000\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \000\000\121\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\117\000\255\255\255\255\
    \117\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\117\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\119\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \118\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\117\000\255\255\255\255\
    \117\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\117\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\119\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\000\000\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \118\000\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\124\000\255\255\255\255\124\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\124\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \126\000\255\255\124\000\000\000\000\000\124\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\124\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\255\255\
    \255\255\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\125\000\255\255\
    \255\255\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \129\000\000\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\000\000\128\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\124\000\255\255\255\255\124\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \124\000\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\126\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\132\000\000\000\
    \000\000\132\000\132\000\132\000\000\000\000\000\000\000\132\000\
    \132\000\000\000\132\000\132\000\132\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\132\000\
    \255\255\132\000\132\000\132\000\132\000\132\000\000\000\000\000\
    \000\000\134\000\000\000\134\000\134\000\134\000\134\000\000\000\
    \000\000\000\000\134\000\134\000\000\000\134\000\134\000\134\000\
    \000\000\000\000\000\000\255\255\125\000\255\255\255\255\255\255\
    \000\000\000\000\134\000\132\000\134\000\134\000\134\000\134\000\
    \134\000\000\000\000\000\000\000\006\000\000\000\000\000\006\000\
    \006\000\006\000\000\000\000\000\000\000\006\000\006\000\000\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\132\000\000\000\132\000\000\000\006\000\134\000\006\000\
    \006\000\006\000\006\000\006\000\000\000\000\000\000\000\146\000\
    \000\000\000\000\146\000\146\000\146\000\000\000\000\000\000\000\
    \146\000\146\000\000\000\146\000\146\000\146\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\134\000\000\000\134\000\000\000\
    \146\000\006\000\146\000\146\000\146\000\146\000\146\000\000\000\
    \000\000\000\000\146\000\000\000\000\000\146\000\146\000\146\000\
    \000\000\000\000\000\000\146\000\146\000\000\000\146\000\146\000\
    \146\000\000\000\000\000\000\000\000\000\000\000\000\000\006\000\
    \000\000\006\000\000\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\000\000\000\000\000\000\146\000\000\000\000\000\
    \146\000\146\000\146\000\000\000\000\000\000\000\146\000\146\000\
    \255\255\146\000\146\000\146\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\146\000\000\000\146\000\000\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\000\000\000\000\000\000\
    \006\000\000\000\000\000\006\000\006\000\006\000\000\000\000\000\
    \000\000\006\000\006\000\000\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\146\000\000\000\146\000\
    \000\000\006\000\146\000\006\000\006\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\006\000\000\000\000\000\006\000\006\000\
    \006\000\000\000\000\000\000\000\006\000\006\000\000\000\006\000\
    \006\000\006\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \146\000\000\000\146\000\000\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\164\000\000\000\
    \164\000\164\000\164\000\164\000\000\000\000\000\000\000\164\000\
    \164\000\000\000\164\000\164\000\164\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\006\000\000\000\006\000\000\000\164\000\
    \006\000\164\000\164\000\164\000\164\000\164\000\000\000\000\000\
    \000\000\164\000\000\000\164\000\164\000\164\000\164\000\000\000\
    \000\000\000\000\164\000\164\000\000\000\164\000\164\000\164\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\006\000\000\000\
    \006\000\000\000\164\000\164\000\164\000\164\000\164\000\164\000\
    \164\000\000\000\000\000\000\000\146\000\000\000\000\000\146\000\
    \146\000\146\000\000\000\000\000\000\000\146\000\146\000\000\000\
    \146\000\146\000\146\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\164\000\000\000\164\000\000\000\146\000\164\000\146\000\
    \146\000\146\000\146\000\146\000\000\000\000\000\000\000\146\000\
    \000\000\000\000\146\000\146\000\146\000\000\000\000\000\000\000\
    \146\000\146\000\000\000\146\000\146\000\146\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\164\000\000\000\164\000\000\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\171\000\000\000\
    \000\000\172\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\146\000\
    \000\000\146\000\000\000\000\000\146\000\000\000\176\000\000\000\
    \000\000\000\000\000\000\174\000\178\000\000\000\177\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\146\000\000\000\146\000\170\000\170\000\170\000\
    \170\000\170\000\170\000\170\000\170\000\170\000\170\000\170\000\
    \170\000\170\000\170\000\170\000\170\000\170\000\170\000\170\000\
    \170\000\170\000\170\000\170\000\170\000\170\000\170\000\000\000\
    \000\000\000\000\000\000\170\000\000\000\170\000\170\000\170\000\
    \170\000\170\000\170\000\170\000\170\000\170\000\170\000\170\000\
    \170\000\170\000\170\000\170\000\170\000\170\000\170\000\170\000\
    \170\000\170\000\170\000\170\000\170\000\170\000\170\000\175\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\000\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\182\000\173\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\185\000\
    \181\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\000\000\000\000\000\000\
    \000\000\184\000\000\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\186\000\255\255\255\255\186\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\186\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\187\000\
    \255\255\186\000\000\000\000\000\186\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \186\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\181\000\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\182\000\
    \000\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\000\000\181\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \186\000\255\255\255\255\186\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\186\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\187\000\255\255\000\000\
    \000\000\199\000\000\000\199\000\000\000\000\000\000\000\000\000\
    \199\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\198\000\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\198\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\199\000\224\000\000\000\
    \000\000\225\000\000\000\199\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\181\000\255\255\255\255\255\255\199\000\
    \197\000\000\000\000\000\199\000\223\000\199\000\223\000\000\000\
    \000\000\196\000\000\000\223\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\243\000\000\000\
    \243\000\243\000\243\000\243\000\243\000\243\000\243\000\243\000\
    \243\000\243\000\243\000\243\000\243\000\243\000\243\000\243\000\
    \243\000\243\000\243\000\243\000\243\000\243\000\243\000\243\000\
    \243\000\243\000\000\000\000\000\242\000\000\000\000\000\000\000\
    \223\000\000\000\000\000\000\000\000\000\000\000\223\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\223\000\000\000\000\000\000\000\223\000\000\000\
    \223\000\220\000\000\000\000\000\221\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\043\000\000\000\000\000\043\000\044\000\
    \046\000\047\000\044\000\046\000\047\000\091\000\138\000\144\000\
    \091\000\138\000\144\000\172\000\192\000\251\000\172\000\192\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\015\000\
    \012\000\016\000\024\000\041\000\016\000\016\000\041\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\089\000\090\000\098\000\098\000\098\000\098\000\012\000\
    \099\000\012\000\041\000\012\000\087\000\087\000\087\000\087\000\
    \087\000\087\000\087\000\087\000\087\000\087\000\097\000\097\000\
    \097\000\097\000\097\000\097\000\097\000\097\000\097\000\097\000\
    \101\000\101\000\101\000\101\000\101\000\101\000\101\000\101\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\012\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\103\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\106\000\109\000\003\000\003\000\003\000\130\000\
    \154\000\026\000\003\000\003\000\026\000\003\000\003\000\003\000\
    \105\000\105\000\105\000\105\000\105\000\105\000\105\000\105\000\
    \105\000\105\000\003\000\155\000\003\000\003\000\003\000\003\000\
    \003\000\159\000\139\000\177\000\004\000\139\000\026\000\004\000\
    \004\000\004\000\178\000\190\000\140\000\004\000\004\000\140\000\
    \004\000\004\000\004\000\191\000\199\000\142\000\141\000\141\000\
    \142\000\174\000\141\000\202\000\174\000\004\000\003\000\004\000\
    \004\000\004\000\004\000\004\000\140\000\206\000\209\000\005\000\
    \220\000\244\000\005\000\005\000\005\000\141\000\245\000\141\000\
    \005\000\005\000\248\000\005\000\005\000\005\000\174\000\197\000\
    \197\000\197\000\197\000\026\000\003\000\250\000\003\000\255\255\
    \005\000\004\000\005\000\005\000\005\000\005\000\005\000\224\000\
    \211\000\216\000\006\000\211\000\216\000\006\000\006\000\006\000\
    \255\255\246\000\212\000\006\000\006\000\212\000\006\000\006\000\
    \006\000\225\000\226\000\247\000\225\000\226\000\224\000\004\000\
    \255\255\004\000\255\255\006\000\005\000\006\000\006\000\006\000\
    \006\000\006\000\212\000\174\000\234\000\007\000\249\000\234\000\
    \007\000\007\000\007\000\255\255\255\255\235\000\007\000\007\000\
    \235\000\007\000\007\000\007\000\240\000\241\000\255\255\240\000\
    \241\000\255\255\005\000\255\255\005\000\255\255\007\000\006\000\
    \007\000\007\000\007\000\007\000\007\000\255\255\255\255\255\255\
    \008\000\255\255\008\000\008\000\008\000\008\000\255\255\255\255\
    \255\255\008\000\008\000\255\255\008\000\008\000\008\000\255\255\
    \255\255\255\255\255\255\255\255\212\000\006\000\255\255\006\000\
    \255\255\008\000\007\000\008\000\008\000\008\000\008\000\008\000\
    \255\255\255\255\255\255\010\000\255\255\255\255\010\000\010\000\
    \010\000\255\255\255\255\255\255\010\000\010\000\255\255\010\000\
    \010\000\010\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \007\000\255\255\007\000\255\255\010\000\008\000\010\000\010\000\
    \010\000\010\000\010\000\255\255\255\255\255\255\255\255\255\255\
    \013\000\255\255\255\255\013\000\013\000\013\000\255\255\026\000\
    \255\255\013\000\013\000\255\255\013\000\013\000\013\000\235\000\
    \255\255\255\255\255\255\008\000\255\255\008\000\255\255\010\000\
    \010\000\013\000\255\255\013\000\013\000\013\000\013\000\013\000\
    \139\000\255\255\255\255\014\000\255\255\255\255\014\000\014\000\
    \014\000\255\255\140\000\255\255\014\000\014\000\255\255\014\000\
    \014\000\014\000\255\255\142\000\255\255\141\000\010\000\174\000\
    \010\000\255\255\255\255\255\255\014\000\013\000\014\000\014\000\
    \014\000\014\000\014\000\255\255\255\255\255\255\017\000\255\255\
    \255\255\017\000\017\000\017\000\255\255\255\255\255\255\017\000\
    \017\000\255\255\017\000\017\000\017\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\013\000\255\255\013\000\255\255\017\000\
    \014\000\017\000\017\000\017\000\017\000\017\000\255\255\255\255\
    \255\255\021\000\255\255\255\255\021\000\021\000\021\000\246\000\
    \212\000\255\255\021\000\021\000\255\255\021\000\021\000\021\000\
    \255\255\247\000\255\255\255\255\255\255\255\255\014\000\255\255\
    \014\000\255\255\021\000\017\000\021\000\021\000\021\000\021\000\
    \021\000\255\255\255\255\255\255\249\000\255\255\255\255\255\255\
    \136\000\255\255\255\255\235\000\255\255\255\255\066\000\255\255\
    \255\255\066\000\066\000\066\000\255\255\255\255\255\255\066\000\
    \066\000\017\000\066\000\017\000\066\000\255\255\021\000\136\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\066\000\
    \022\000\255\255\066\000\066\000\066\000\066\000\255\255\136\000\
    \136\000\136\000\136\000\136\000\136\000\136\000\136\000\136\000\
    \136\000\255\255\255\255\255\255\021\000\255\255\021\000\022\000\
    \022\000\255\255\022\000\022\000\022\000\022\000\255\255\255\255\
    \255\255\022\000\022\000\066\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\255\255\022\000\022\000\022\000\022\000\022\000\
    \255\255\255\255\255\255\023\000\255\255\255\255\023\000\023\000\
    \023\000\066\000\255\255\023\000\023\000\023\000\255\255\023\000\
    \023\000\023\000\198\000\198\000\198\000\198\000\198\000\198\000\
    \198\000\198\000\198\000\198\000\023\000\022\000\023\000\023\000\
    \023\000\023\000\023\000\255\255\255\255\025\000\201\000\201\000\
    \201\000\201\000\201\000\201\000\201\000\201\000\201\000\201\000\
    \204\000\204\000\204\000\204\000\204\000\204\000\204\000\204\000\
    \255\255\255\255\255\255\022\000\025\000\022\000\255\255\114\000\
    \023\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\255\255\114\000\255\255\023\000\025\000\
    \023\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\255\255\025\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\255\255\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\255\255\255\255\
    \255\255\255\255\028\000\255\255\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\255\255\255\255\
    \255\255\255\255\028\000\255\255\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\255\255\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\255\255\029\000\
    \255\255\255\255\029\000\255\255\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\255\255\029\000\
    \255\255\255\255\029\000\255\255\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\028\000\255\255\255\255\
    \255\255\255\255\255\255\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\255\255\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\096\000\096\000\096\000\096\000\096\000\
    \096\000\096\000\096\000\096\000\096\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\096\000\096\000\096\000\096\000\
    \096\000\096\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\255\255\255\255\030\000\030\000\
    \030\000\030\000\255\255\030\000\096\000\096\000\096\000\096\000\
    \096\000\096\000\227\000\227\000\227\000\227\000\227\000\227\000\
    \227\000\227\000\227\000\227\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\029\000\030\000\030\000\
    \030\000\030\000\030\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\255\255\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\031\000\031\000\
    \031\000\031\000\255\255\031\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\031\000\031\000\
    \031\000\031\000\031\000\030\000\255\255\255\255\255\255\255\255\
    \255\255\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \255\255\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\135\000\135\000\255\255\255\255\135\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\135\000\255\255\135\000\032\000\032\000\032\000\032\000\
    \255\255\032\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\031\000\032\000\032\000\032\000\032\000\
    \032\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \255\255\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\033\000\033\000\033\000\033\000\
    \115\000\033\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\033\000\033\000\033\000\033\000\
    \033\000\032\000\255\255\255\255\255\255\255\255\255\255\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\255\255\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\255\255\
    \255\255\135\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\196\000\196\000\196\000\196\000\196\000\196\000\255\255\
    \255\255\255\255\034\000\034\000\034\000\034\000\255\255\034\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\034\000\255\255\255\255\
    \255\255\196\000\196\000\196\000\196\000\196\000\196\000\255\255\
    \255\255\033\000\034\000\034\000\034\000\034\000\034\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\255\255\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\036\000\255\255\036\000\036\000\036\000\036\000\
    \255\255\255\255\255\255\036\000\036\000\255\255\036\000\036\000\
    \036\000\255\255\035\000\035\000\035\000\035\000\255\255\035\000\
    \255\255\255\255\255\255\036\000\035\000\036\000\036\000\036\000\
    \036\000\036\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\035\000\035\000\035\000\035\000\035\000\034\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\036\000\
    \036\000\255\255\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\255\255\036\000\039\000\036\000\
    \255\255\039\000\039\000\039\000\255\255\255\255\255\255\039\000\
    \039\000\255\255\039\000\039\000\039\000\208\000\208\000\208\000\
    \208\000\208\000\208\000\208\000\208\000\208\000\208\000\039\000\
    \255\255\039\000\039\000\039\000\039\000\039\000\208\000\208\000\
    \208\000\208\000\208\000\208\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\039\000\039\000\255\255\255\255\208\000\208\000\
    \208\000\208\000\208\000\208\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\035\000\
    \255\255\039\000\039\000\039\000\255\255\255\255\255\255\255\255\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \255\255\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\037\000\255\255\037\000\037\000\037\000\037\000\255\255\
    \255\255\255\255\037\000\037\000\255\255\037\000\037\000\037\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\037\000\255\255\037\000\037\000\037\000\037\000\
    \037\000\255\255\237\000\255\255\237\000\237\000\237\000\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\237\000\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\237\000\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\037\000\037\000\
    \237\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\255\255\037\000\040\000\037\000\255\255\
    \040\000\040\000\040\000\255\255\255\255\255\255\040\000\040\000\
    \255\255\040\000\040\000\040\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\040\000\255\255\
    \040\000\040\000\040\000\040\000\040\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\255\255\
    \255\255\255\255\040\000\119\000\255\255\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\255\255\
    \040\000\255\255\040\000\255\255\255\255\255\255\255\255\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\255\255\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\255\255\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\255\255\255\255\255\255\255\255\255\255\049\000\255\255\
    \255\255\049\000\049\000\049\000\255\255\255\255\255\255\049\000\
    \049\000\255\255\049\000\049\000\049\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\038\000\038\000\038\000\038\000\049\000\
    \038\000\049\000\049\000\049\000\049\000\049\000\255\255\255\255\
    \255\255\054\000\255\255\255\255\054\000\054\000\054\000\255\255\
    \255\255\255\255\054\000\054\000\255\255\054\000\054\000\054\000\
    \255\255\255\255\255\255\038\000\038\000\038\000\038\000\038\000\
    \255\255\255\255\054\000\049\000\054\000\054\000\054\000\054\000\
    \054\000\255\255\255\255\255\255\055\000\255\255\055\000\055\000\
    \055\000\055\000\255\255\255\255\255\255\055\000\055\000\255\255\
    \055\000\055\000\055\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\049\000\255\255\049\000\255\255\055\000\054\000\055\000\
    \055\000\055\000\055\000\055\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\054\000\255\255\054\000\255\255\
    \122\000\055\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\255\255\122\000\255\255\055\000\
    \255\255\055\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \038\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \255\255\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\056\000\056\000\056\000\056\000\
    \126\000\056\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\056\000\056\000\056\000\056\000\
    \056\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \255\255\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\057\000\057\000\057\000\057\000\
    \185\000\057\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\185\000\185\000\185\000\185\000\
    \185\000\185\000\185\000\185\000\057\000\057\000\057\000\057\000\
    \057\000\056\000\255\255\255\255\255\255\255\255\255\255\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\255\255\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\060\000\060\000\060\000\060\000\255\255\060\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\057\000\060\000\060\000\060\000\060\000\060\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\255\255\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\061\000\061\000\061\000\061\000\187\000\061\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\061\000\061\000\061\000\061\000\061\000\060\000\
    \255\255\255\255\255\255\255\255\255\255\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\255\255\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\229\000\229\000\229\000\
    \229\000\229\000\229\000\229\000\229\000\229\000\229\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\229\000\229\000\
    \229\000\229\000\229\000\229\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \064\000\064\000\064\000\064\000\255\255\064\000\229\000\229\000\
    \229\000\229\000\229\000\229\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\064\000\255\255\255\255\255\255\255\255\255\255\061\000\
    \064\000\064\000\064\000\064\000\064\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\255\255\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\231\000\231\000\231\000\
    \231\000\231\000\231\000\231\000\231\000\231\000\231\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\231\000\231\000\
    \231\000\231\000\231\000\231\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \065\000\065\000\065\000\065\000\255\255\065\000\231\000\231\000\
    \231\000\231\000\231\000\231\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \065\000\065\000\065\000\065\000\065\000\064\000\255\255\255\255\
    \255\255\255\255\255\255\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\255\255\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\067\000\067\000\
    \067\000\067\000\255\255\067\000\255\255\255\255\255\255\067\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\065\000\067\000\067\000\
    \067\000\067\000\067\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\255\255\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\255\255\255\255\255\255\255\255\
    \255\255\069\000\255\255\255\255\069\000\069\000\069\000\255\255\
    \255\255\255\255\069\000\069\000\255\255\069\000\255\255\069\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\068\000\068\000\
    \068\000\068\000\069\000\068\000\255\255\069\000\069\000\069\000\
    \069\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\068\000\068\000\
    \068\000\068\000\068\000\067\000\255\255\129\000\069\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\255\255\129\000\255\255\069\000\255\255\255\255\255\255\
    \255\255\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \255\255\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\068\000\070\000\070\000\070\000\070\000\
    \255\255\070\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\070\000\070\000\070\000\070\000\
    \070\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \255\255\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\232\000\232\000\232\000\232\000\232\000\232\000\
    \232\000\232\000\232\000\232\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\232\000\232\000\232\000\232\000\232\000\
    \232\000\255\255\255\255\255\255\071\000\071\000\071\000\071\000\
    \255\255\071\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\232\000\232\000\232\000\232\000\232\000\
    \232\000\255\255\255\255\255\255\071\000\071\000\071\000\071\000\
    \071\000\070\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\232\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\255\255\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\255\255\255\255\255\255\255\255\
    \072\000\255\255\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\255\255\255\255\255\255\255\255\
    \072\000\071\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\255\255\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\073\000\073\000\
    \073\000\073\000\255\255\073\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\073\000\073\000\
    \073\000\073\000\073\000\072\000\255\255\255\255\255\255\255\255\
    \255\255\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \255\255\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\074\000\074\000\074\000\074\000\
    \255\255\074\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\073\000\074\000\074\000\074\000\074\000\
    \074\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \255\255\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\255\255\255\255\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\075\000\075\000\075\000\075\000\
    \255\255\075\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\075\000\075\000\075\000\075\000\
    \075\000\074\000\255\255\255\255\255\255\255\255\255\255\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\255\255\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\076\000\076\000\076\000\076\000\255\255\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\075\000\076\000\076\000\076\000\076\000\076\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\255\255\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\076\000\
    \112\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\112\000\112\000\112\000\112\000\
    \112\000\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
    \112\000\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
    \112\000\112\000\112\000\112\000\112\000\112\000\255\255\255\255\
    \255\255\255\255\112\000\255\255\112\000\112\000\112\000\112\000\
    \112\000\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
    \112\000\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
    \112\000\112\000\112\000\112\000\112\000\112\000\255\255\255\255\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\077\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\255\255\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\079\000\079\000\079\000\079\000\255\255\
    \079\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\079\000\079\000\079\000\079\000\079\000\
    \078\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\255\255\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\255\255\255\255\255\255\255\255\255\255\
    \079\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\255\255\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\082\000\082\000\082\000\
    \082\000\255\255\082\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\082\000\082\000\082\000\
    \082\000\082\000\080\000\255\255\255\255\255\255\255\255\255\255\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\255\255\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\083\000\083\000\083\000\083\000\255\255\
    \083\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\082\000\083\000\083\000\083\000\083\000\083\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\255\255\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \255\255\255\255\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\255\255\255\255\255\255\255\255\255\255\255\255\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\255\255\255\255\255\255\255\255\255\255\255\255\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \083\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\255\255\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\255\255\255\255\255\255\255\255\255\255\
    \084\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\255\255\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\255\255\175\000\255\255\255\255\255\255\
    \255\255\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\085\000\088\000\255\255\088\000\117\000\255\255\
    \255\255\117\000\088\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\117\000\175\000\255\255\
    \175\000\175\000\175\000\175\000\175\000\175\000\175\000\175\000\
    \175\000\175\000\175\000\175\000\175\000\175\000\175\000\175\000\
    \175\000\175\000\175\000\175\000\175\000\175\000\175\000\175\000\
    \175\000\175\000\255\255\175\000\255\255\255\255\255\255\088\000\
    \255\255\255\255\255\255\255\255\255\255\088\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\088\000\088\000\255\255\255\255\088\000\255\255\088\000\
    \255\255\255\255\255\255\088\000\117\000\255\255\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \255\255\117\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\086\000\255\255\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\255\255\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\116\000\
    \116\000\116\000\116\000\088\000\116\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\116\000\
    \116\000\116\000\116\000\116\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\255\255\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\120\000\
    \120\000\120\000\120\000\255\255\120\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\120\000\
    \120\000\120\000\120\000\120\000\116\000\255\255\255\255\255\255\
    \255\255\255\255\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\255\255\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\124\000\255\255\255\255\124\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\124\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\123\000\123\000\123\000\
    \123\000\255\255\123\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\120\000\123\000\123\000\123\000\
    \123\000\123\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \124\000\255\255\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\124\000\124\000\124\000\124\000\
    \124\000\124\000\124\000\124\000\255\255\124\000\255\255\255\255\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\255\255\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\255\255\255\255\255\255\255\255\255\255\132\000\255\255\
    \255\255\132\000\132\000\132\000\255\255\255\255\255\255\132\000\
    \132\000\255\255\132\000\132\000\132\000\255\255\255\255\255\255\
    \255\255\255\255\123\000\127\000\127\000\127\000\127\000\132\000\
    \127\000\132\000\132\000\132\000\132\000\132\000\255\255\255\255\
    \255\255\134\000\255\255\134\000\134\000\134\000\134\000\255\255\
    \255\255\255\255\134\000\134\000\255\255\134\000\134\000\134\000\
    \255\255\255\255\255\255\127\000\127\000\127\000\127\000\127\000\
    \255\255\255\255\134\000\132\000\134\000\134\000\134\000\134\000\
    \134\000\255\255\255\255\255\255\145\000\255\255\255\255\145\000\
    \145\000\145\000\255\255\255\255\255\255\145\000\145\000\255\255\
    \145\000\145\000\145\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\132\000\255\255\132\000\255\255\145\000\134\000\145\000\
    \145\000\145\000\145\000\145\000\255\255\255\255\255\255\146\000\
    \255\255\255\255\146\000\146\000\146\000\255\255\255\255\255\255\
    \146\000\146\000\255\255\146\000\146\000\146\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\134\000\255\255\134\000\255\255\
    \146\000\145\000\146\000\146\000\146\000\146\000\146\000\255\255\
    \255\255\255\255\147\000\255\255\255\255\147\000\147\000\147\000\
    \255\255\255\255\255\255\147\000\147\000\255\255\147\000\147\000\
    \147\000\255\255\255\255\255\255\255\255\255\255\255\255\145\000\
    \255\255\145\000\255\255\147\000\146\000\147\000\147\000\147\000\
    \147\000\147\000\255\255\255\255\255\255\148\000\255\255\255\255\
    \148\000\148\000\148\000\255\255\255\255\255\255\148\000\148\000\
    \127\000\148\000\148\000\148\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\146\000\255\255\146\000\255\255\148\000\147\000\
    \148\000\148\000\148\000\148\000\148\000\255\255\255\255\255\255\
    \153\000\255\255\255\255\153\000\153\000\153\000\255\255\255\255\
    \255\255\153\000\153\000\255\255\153\000\153\000\153\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\147\000\255\255\147\000\
    \255\255\153\000\148\000\153\000\153\000\153\000\153\000\153\000\
    \255\255\255\255\255\255\163\000\255\255\255\255\163\000\163\000\
    \163\000\255\255\255\255\255\255\163\000\163\000\255\255\163\000\
    \163\000\163\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \148\000\255\255\148\000\255\255\163\000\153\000\163\000\163\000\
    \163\000\163\000\163\000\255\255\255\255\255\255\164\000\255\255\
    \164\000\164\000\164\000\164\000\255\255\255\255\255\255\164\000\
    \164\000\255\255\164\000\164\000\164\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\153\000\255\255\153\000\255\255\164\000\
    \163\000\164\000\164\000\164\000\164\000\164\000\255\255\255\255\
    \255\255\165\000\255\255\165\000\165\000\165\000\165\000\255\255\
    \255\255\255\255\165\000\165\000\255\255\165\000\165\000\165\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\163\000\255\255\
    \163\000\255\255\165\000\164\000\165\000\165\000\165\000\165\000\
    \165\000\255\255\255\255\255\255\166\000\255\255\255\255\166\000\
    \166\000\166\000\255\255\255\255\255\255\166\000\166\000\255\255\
    \166\000\166\000\166\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\164\000\255\255\164\000\255\255\166\000\165\000\166\000\
    \166\000\166\000\166\000\166\000\255\255\255\255\255\255\167\000\
    \255\255\255\255\167\000\167\000\167\000\255\255\255\255\255\255\
    \167\000\167\000\255\255\167\000\167\000\167\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\165\000\255\255\165\000\255\255\
    \167\000\166\000\167\000\167\000\167\000\167\000\167\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\168\000\255\255\
    \255\255\168\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\166\000\
    \255\255\166\000\255\255\255\255\167\000\255\255\168\000\255\255\
    \255\255\255\255\255\255\168\000\168\000\255\255\168\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\167\000\255\255\167\000\168\000\168\000\168\000\
    \168\000\168\000\168\000\168\000\168\000\168\000\168\000\168\000\
    \168\000\168\000\168\000\168\000\168\000\168\000\168\000\168\000\
    \168\000\168\000\168\000\168\000\168\000\168\000\168\000\255\255\
    \255\255\255\255\255\255\168\000\255\255\168\000\168\000\168\000\
    \168\000\168\000\168\000\168\000\168\000\168\000\168\000\168\000\
    \168\000\168\000\168\000\168\000\168\000\168\000\168\000\168\000\
    \168\000\168\000\168\000\168\000\168\000\168\000\168\000\168\000\
    \170\000\170\000\170\000\170\000\170\000\170\000\170\000\170\000\
    \170\000\170\000\170\000\170\000\170\000\170\000\170\000\170\000\
    \170\000\170\000\170\000\170\000\170\000\170\000\170\000\170\000\
    \170\000\170\000\170\000\170\000\170\000\170\000\170\000\170\000\
    \170\000\170\000\170\000\170\000\170\000\170\000\170\000\255\255\
    \170\000\170\000\170\000\170\000\170\000\170\000\170\000\170\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\170\000\170\000\170\000\170\000\170\000\170\000\
    \170\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\170\000\170\000\170\000\170\000\255\255\
    \170\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\170\000\170\000\170\000\170\000\170\000\
    \255\255\255\255\255\255\182\000\168\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\183\000\
    \182\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\183\000\183\000\183\000\183\000\183\000\
    \183\000\183\000\183\000\183\000\183\000\183\000\183\000\183\000\
    \183\000\183\000\183\000\183\000\183\000\183\000\183\000\183\000\
    \183\000\183\000\183\000\183\000\183\000\255\255\255\255\255\255\
    \255\255\183\000\255\255\183\000\183\000\183\000\183\000\183\000\
    \183\000\183\000\183\000\183\000\183\000\183\000\183\000\183\000\
    \183\000\183\000\183\000\183\000\183\000\183\000\183\000\183\000\
    \183\000\183\000\183\000\183\000\183\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \170\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \255\255\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\186\000\255\255\255\255\186\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \186\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\184\000\184\000\184\000\184\000\
    \255\255\184\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\184\000\184\000\184\000\184\000\
    \184\000\255\255\255\255\255\255\255\255\255\255\255\255\186\000\
    \255\255\186\000\186\000\186\000\186\000\186\000\186\000\186\000\
    \186\000\186\000\186\000\186\000\186\000\186\000\186\000\186\000\
    \186\000\186\000\186\000\186\000\186\000\186\000\186\000\186\000\
    \186\000\186\000\186\000\255\255\186\000\255\255\255\255\188\000\
    \188\000\188\000\188\000\188\000\188\000\188\000\188\000\188\000\
    \188\000\188\000\188\000\188\000\188\000\188\000\188\000\188\000\
    \188\000\188\000\188\000\188\000\188\000\188\000\188\000\188\000\
    \188\000\188\000\188\000\188\000\188\000\188\000\188\000\188\000\
    \188\000\188\000\188\000\188\000\188\000\188\000\255\255\188\000\
    \188\000\188\000\188\000\188\000\188\000\188\000\188\000\255\255\
    \255\255\189\000\255\255\189\000\255\255\255\255\255\255\255\255\
    \189\000\188\000\188\000\188\000\188\000\188\000\188\000\188\000\
    \255\255\189\000\189\000\189\000\189\000\189\000\189\000\189\000\
    \189\000\189\000\189\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\184\000\188\000\188\000\188\000\188\000\255\255\188\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\189\000\217\000\255\255\
    \255\255\217\000\255\255\189\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\188\000\188\000\188\000\188\000\188\000\189\000\
    \189\000\255\255\255\255\189\000\217\000\189\000\217\000\255\255\
    \255\255\189\000\255\255\217\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\217\000\217\000\217\000\217\000\
    \217\000\217\000\217\000\217\000\217\000\217\000\243\000\255\255\
    \243\000\243\000\243\000\243\000\243\000\243\000\243\000\243\000\
    \243\000\243\000\243\000\243\000\243\000\243\000\243\000\243\000\
    \243\000\243\000\243\000\243\000\243\000\243\000\243\000\243\000\
    \243\000\243\000\255\255\255\255\243\000\255\255\255\255\255\255\
    \217\000\255\255\255\255\255\255\255\255\255\255\217\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\217\000\255\255\255\255\255\255\217\000\255\255\
    \217\000\217\000\255\255\255\255\217\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\188\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\217\000";
  Lexing.lex_base_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\058\000\181\000\001\000\000\000\181\000\
    \053\001\016\000\000\000\187\001\002\000\000\000\053\001\059\002\
    \024\000\000\000\000\000\000\000\000\000\000\000\000\000\204\001\
    \115\002\034\000\000\000\000\000\229\001\002\000\000\000\049\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\002\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\005\000\006\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \061\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_default_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\007\000\000\000\000\000\000\000\
    \007\000\000\000\000\000\010\000\000\000\000\000\000\000\010\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\041\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\013\000\021\000\058\000\013\000\021\000\058\000\
    \058\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\013\000\021\000\001\000\044\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\000\000\000\000\000\000\000\000\007\000\
    \000\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\000\000\000\000\000\000\
    \000\000\010\000\000\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\013\000\000\000\
    \000\000\013\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\013\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \000\000\000\000\000\000\000\000\007\000\000\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\013\000\000\000\
    \000\000\013\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\013\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \000\000\000\000\000\000\000\000\010\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\021\000\000\000\000\000\021\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\021\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\029\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\021\000\000\000\000\000\021\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\021\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\001\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\001\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_check_code =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\022\000\117\000\124\000\217\000\117\000\124\000\225\000\
    \226\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \022\000\117\000\124\000\000\000\141\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
    \112\000\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
    \112\000\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
    \112\000\112\000\112\000\255\255\255\255\255\255\255\255\112\000\
    \255\255\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
    \112\000\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
    \112\000\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
    \112\000\112\000\112\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\255\255\255\255\255\255\
    \255\255\115\000\255\255\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\255\255\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \116\000\116\000\116\000\116\000\119\000\116\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \116\000\116\000\116\000\116\000\116\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\255\255\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \120\000\120\000\120\000\120\000\126\000\120\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \126\000\126\000\126\000\126\000\126\000\126\000\126\000\126\000\
    \120\000\120\000\120\000\120\000\120\000\116\000\255\255\255\255\
    \255\255\255\255\255\255\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\255\255\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\255\255\255\255\255\255\135\000\140\000\
    \255\255\255\255\140\000\255\255\255\255\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\255\255\140\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\123\000\123\000\
    \123\000\123\000\255\255\123\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\120\000\123\000\123\000\
    \123\000\123\000\123\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\255\255\127\000\127\000\127\000\127\000\127\000\
    \127\000\127\000\127\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\127\000\127\000\127\000\
    \127\000\127\000\127\000\127\000\136\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\136\000\255\255\255\255\127\000\127\000\
    \127\000\127\000\255\255\127\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\136\000\136\000\136\000\136\000\136\000\
    \136\000\136\000\136\000\136\000\136\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\127\000\127\000\
    \127\000\127\000\127\000\123\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\140\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\127\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255";
  Lexing.lex_code =
   "\255\004\255\255\009\255\255\005\255\255\007\255\255\006\255\255\
    \000\005\001\006\255\008\255\255\000\007\001\008\255\011\255\010\
    \255\255\003\255\000\004\001\009\255\011\255\255\010\255\011\255\
    \255\000\004\001\009\003\010\002\011\255\001\255\255\000\001\255\
    ";
}

let rec token state lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 12 (-1); __ocaml_lex_token_rec state lexbuf 0
and __ocaml_lex_token_rec state lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 396 "src/ocaml/preprocess/lexer_raw.mll"
             bs
# 2733 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 396 "src/ocaml/preprocess/lexer_raw.mll"
                         (
      match state.preprocessor with
      | None -> fail lexbuf (Illegal_character bs)
      | Some _ ->
        update_loc lexbuf None 1 false 0;
        token state lexbuf )
# 2742 "src/ocaml/preprocess/lexer_raw.ml"

  | 1 ->
# 403 "src/ocaml/preprocess/lexer_raw.mll"
      ( update_loc lexbuf None 1 false 0;
        match state.preprocessor with
        | None -> token state lexbuf
        | Some _ -> return EOL
      )
# 2751 "src/ocaml/preprocess/lexer_raw.ml"

  | 2 ->
# 409 "src/ocaml/preprocess/lexer_raw.mll"
      ( token state lexbuf )
# 2756 "src/ocaml/preprocess/lexer_raw.ml"

  | 3 ->
# 411 "src/ocaml/preprocess/lexer_raw.mll"
      ( return DOTLESS )
# 2761 "src/ocaml/preprocess/lexer_raw.ml"

  | 4 ->
# 413 "src/ocaml/preprocess/lexer_raw.mll"
      ( return (keyword_or state (Lexing.lexeme lexbuf) (INFIXOP0 ">.")) )
# 2766 "src/ocaml/preprocess/lexer_raw.ml"

  | 5 ->
# 415 "src/ocaml/preprocess/lexer_raw.mll"
      ( return (keyword_or state (Lexing.lexeme lexbuf) DOTTILDE) )
# 2771 "src/ocaml/preprocess/lexer_raw.ml"

  | 6 ->
# 417 "src/ocaml/preprocess/lexer_raw.mll"
      ( return UNDERSCORE )
# 2776 "src/ocaml/preprocess/lexer_raw.ml"

  | 7 ->
# 419 "src/ocaml/preprocess/lexer_raw.mll"
      ( return TILDE )
# 2781 "src/ocaml/preprocess/lexer_raw.ml"

  | 8 ->
let
# 425 "src/ocaml/preprocess/lexer_raw.mll"
                                  name
# 2787 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_curr_pos + -1) in
# 426 "src/ocaml/preprocess/lexer_raw.mll"
      ( lABEL (check_label_name lexbuf name) )
# 2791 "src/ocaml/preprocess/lexer_raw.ml"

  | 9 ->
let
# 427 "src/ocaml/preprocess/lexer_raw.mll"
                                                name
# 2797 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_curr_pos + -1) in
# 428 "src/ocaml/preprocess/lexer_raw.mll"
      ( warn_latin1 lexbuf;
        return (LABEL name) )
# 2802 "src/ocaml/preprocess/lexer_raw.ml"

  | 10 ->
# 431 "src/ocaml/preprocess/lexer_raw.mll"
      ( return QUESTION )
# 2807 "src/ocaml/preprocess/lexer_raw.ml"

  | 11 ->
let
# 432 "src/ocaml/preprocess/lexer_raw.mll"
                                  name
# 2813 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_curr_pos + -1) in
# 433 "src/ocaml/preprocess/lexer_raw.mll"
      ( oPTLABEL (check_label_name lexbuf name) )
# 2817 "src/ocaml/preprocess/lexer_raw.ml"

  | 12 ->
let
# 434 "src/ocaml/preprocess/lexer_raw.mll"
                                                name
# 2823 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_curr_pos + -1) in
# 435 "src/ocaml/preprocess/lexer_raw.mll"
      ( warn_latin1 lexbuf; return (OPTLABEL name) )
# 2827 "src/ocaml/preprocess/lexer_raw.ml"

  | 13 ->
let
# 436 "src/ocaml/preprocess/lexer_raw.mll"
                             name
# 2833 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 437 "src/ocaml/preprocess/lexer_raw.mll"
    ( return (try Hashtbl.find state.keywords name
              with Not_found ->
              try Hashtbl.find keyword_table name
              with Not_found ->
                LIDENT name) )
# 2841 "src/ocaml/preprocess/lexer_raw.ml"

  | 14 ->
let
# 442 "src/ocaml/preprocess/lexer_raw.mll"
                                           name
# 2847 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 443 "src/ocaml/preprocess/lexer_raw.mll"
      ( warn_latin1 lexbuf; return (LIDENT name) )
# 2851 "src/ocaml/preprocess/lexer_raw.ml"

  | 15 ->
let
# 444 "src/ocaml/preprocess/lexer_raw.mll"
                             name
# 2857 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 445 "src/ocaml/preprocess/lexer_raw.mll"
    ( (* Capitalized keywords for OUnit *)
      return (try Hashtbl.find state.keywords name
              with Not_found ->
              try Hashtbl.find keyword_table name
              with Not_found ->
                UIDENT name) )
# 2866 "src/ocaml/preprocess/lexer_raw.ml"

  | 16 ->
let
# 451 "src/ocaml/preprocess/lexer_raw.mll"
                                           name
# 2872 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 452 "src/ocaml/preprocess/lexer_raw.mll"
    ( warn_latin1 lexbuf; return (UIDENT name) )
# 2876 "src/ocaml/preprocess/lexer_raw.ml"

  | 17 ->
let
# 453 "src/ocaml/preprocess/lexer_raw.mll"
                   lit
# 2882 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 453 "src/ocaml/preprocess/lexer_raw.mll"
                       ( return (INT (lit, None)) )
# 2886 "src/ocaml/preprocess/lexer_raw.ml"

  | 18 ->
let
# 454 "src/ocaml/preprocess/lexer_raw.mll"
                    lit
# 2892 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_curr_pos + -1)
and
# 454 "src/ocaml/preprocess/lexer_raw.mll"
                                              modif
# 2897 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_curr_pos + -1) in
# 455 "src/ocaml/preprocess/lexer_raw.mll"
    ( return (INT (lit, Some modif)) )
# 2901 "src/ocaml/preprocess/lexer_raw.ml"

  | 19 ->
let
# 456 "src/ocaml/preprocess/lexer_raw.mll"
                                         lit
# 2907 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 457 "src/ocaml/preprocess/lexer_raw.mll"
    ( return (FLOAT (lit, None)) )
# 2911 "src/ocaml/preprocess/lexer_raw.ml"

  | 20 ->
let
# 458 "src/ocaml/preprocess/lexer_raw.mll"
                                          lit
# 2917 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_curr_pos + -1)
and
# 458 "src/ocaml/preprocess/lexer_raw.mll"
                                                                    modif
# 2922 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_curr_pos + -1) in
# 459 "src/ocaml/preprocess/lexer_raw.mll"
    ( return (FLOAT (lit, Some modif)) )
# 2926 "src/ocaml/preprocess/lexer_raw.ml"

  | 21 ->
let
# 460 "src/ocaml/preprocess/lexer_raw.mll"
                                                                    invalid
# 2932 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 461 "src/ocaml/preprocess/lexer_raw.mll"
    ( fail lexbuf (Invalid_literal invalid) )
# 2936 "src/ocaml/preprocess/lexer_raw.ml"

  | 22 ->
# 463 "src/ocaml/preprocess/lexer_raw.mll"
      ( wrap_string_lexer string state lexbuf >>= fun (str, loc) ->
        return (STRING (str, loc, None)) )
# 2942 "src/ocaml/preprocess/lexer_raw.ml"

  | 23 ->
# 466 "src/ocaml/preprocess/lexer_raw.mll"
      ( wrap_string_lexer string state lexbuf >>= fun (str, loc) ->
        return (STRING (str, loc, None)) )
# 2948 "src/ocaml/preprocess/lexer_raw.ml"

  | 24 ->
let
# 468 "src/ocaml/preprocess/lexer_raw.mll"
                       delim
# 2954 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_curr_pos + -1) in
# 469 "src/ocaml/preprocess/lexer_raw.mll"
      ( wrap_string_lexer (quoted_string delim) state lexbuf
        >>= fun (str, loc) ->
        return (STRING (str, loc, Some delim)) )
# 2960 "src/ocaml/preprocess/lexer_raw.ml"

  | 25 ->
let
# 472 "src/ocaml/preprocess/lexer_raw.mll"
                          id
# 2966 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 2) (lexbuf.Lexing.lex_curr_pos + -1) in
# 473 "src/ocaml/preprocess/lexer_raw.mll"
      ( let orig_loc = Location.curr lexbuf in
        wrap_string_lexer (quoted_string "") state lexbuf
        >>= fun (str, loc) ->
        let idloc = compute_quoted_string_idloc orig_loc 2 id in
        return (QUOTED_STRING_EXPR (id, idloc, str, loc, Some "")) )
# 2974 "src/ocaml/preprocess/lexer_raw.ml"

  | 26 ->
let
# 478 "src/ocaml/preprocess/lexer_raw.mll"
                          id
# 2980 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 2) lexbuf.Lexing.lex_mem.(0)
and
# 478 "src/ocaml/preprocess/lexer_raw.mll"
                                                    delim
# 2985 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) (lexbuf.Lexing.lex_curr_pos + -1) in
# 479 "src/ocaml/preprocess/lexer_raw.mll"
      ( let orig_loc = Location.curr lexbuf in
        wrap_string_lexer (quoted_string delim) state lexbuf
        >>= fun (str, loc) ->
        let idloc = compute_quoted_string_idloc orig_loc 2 id in
        return (QUOTED_STRING_EXPR (id, idloc, str, loc, Some delim)) )
# 2993 "src/ocaml/preprocess/lexer_raw.ml"

  | 27 ->
let
# 484 "src/ocaml/preprocess/lexer_raw.mll"
                           id
# 2999 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 3) (lexbuf.Lexing.lex_curr_pos + -1) in
# 485 "src/ocaml/preprocess/lexer_raw.mll"
      ( let orig_loc = Location.curr lexbuf in
        wrap_string_lexer (quoted_string "") state lexbuf
        >>= fun (str, loc) ->
        let idloc = compute_quoted_string_idloc orig_loc 3 id in
        return (QUOTED_STRING_ITEM (id, idloc, str, loc, Some "")) )
# 3007 "src/ocaml/preprocess/lexer_raw.ml"

  | 28 ->
let
# 490 "src/ocaml/preprocess/lexer_raw.mll"
                           id
# 3013 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 3) lexbuf.Lexing.lex_mem.(0)
and
# 490 "src/ocaml/preprocess/lexer_raw.mll"
                                                     delim
# 3018 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) (lexbuf.Lexing.lex_curr_pos + -1) in
# 491 "src/ocaml/preprocess/lexer_raw.mll"
      ( let orig_loc = Location.curr lexbuf in
        wrap_string_lexer (quoted_string delim) state lexbuf
        >>= fun (str, loc) ->
        let idloc = compute_quoted_string_idloc orig_loc 3 id in
        return (QUOTED_STRING_ITEM (id, idloc, str, loc, Some delim)) )
# 3026 "src/ocaml/preprocess/lexer_raw.ml"

  | 29 ->
# 497 "src/ocaml/preprocess/lexer_raw.mll"
    ( update_loc lexbuf None 1 false 1;
      (* newline is ('\013'* '\010') *)
      return (CHAR '\n') )
# 3033 "src/ocaml/preprocess/lexer_raw.ml"

  | 30 ->
let
# 500 "src/ocaml/preprocess/lexer_raw.mll"
                                         c
# 3039 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 501 "src/ocaml/preprocess/lexer_raw.mll"
    ( return (CHAR c) )
# 3043 "src/ocaml/preprocess/lexer_raw.ml"

  | 31 ->
let
# 502 "src/ocaml/preprocess/lexer_raw.mll"
                                                    c
# 3049 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2) in
# 503 "src/ocaml/preprocess/lexer_raw.mll"
    ( return (CHAR (char_for_backslash c)) )
# 3053 "src/ocaml/preprocess/lexer_raw.ml"

  | 32 ->
# 505 "src/ocaml/preprocess/lexer_raw.mll"
    ( char_for_octal_code state lexbuf 3 >>= fun c -> return (CHAR c) )
# 3058 "src/ocaml/preprocess/lexer_raw.ml"

  | 33 ->
# 507 "src/ocaml/preprocess/lexer_raw.mll"
    ( char_for_decimal_code state lexbuf 2 >>= fun c -> return (CHAR c) )
# 3063 "src/ocaml/preprocess/lexer_raw.ml"

  | 34 ->
# 509 "src/ocaml/preprocess/lexer_raw.mll"
    ( return (CHAR (char_for_hexadecimal_code lexbuf 3)) )
# 3068 "src/ocaml/preprocess/lexer_raw.ml"

  | 35 ->
let
# 510 "src/ocaml/preprocess/lexer_raw.mll"
                    esc
# 3074 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_start_pos + 3) in
# 511 "src/ocaml/preprocess/lexer_raw.mll"
      ( fail lexbuf (Illegal_escape (esc, None)) )
# 3078 "src/ocaml/preprocess/lexer_raw.ml"

  | 36 ->
# 513 "src/ocaml/preprocess/lexer_raw.mll"
      ( let start_loc = Location.curr lexbuf in
        state.comment_start_loc <- [start_loc];
        Buffer.reset state.buffer;
        comment state lexbuf >>= fun end_loc ->
        let s = Buffer.contents state.buffer in
        Buffer.reset state.buffer;
        return (COMMENT (s, { start_loc with
                              Location.loc_end = end_loc.Location.loc_end }))
      )
# 3091 "src/ocaml/preprocess/lexer_raw.ml"

  | 37 ->
# 523 "src/ocaml/preprocess/lexer_raw.mll"
      ( let loc = Location.curr lexbuf in
        Location.prerr_warning loc Warnings.Comment_start;
        state.comment_start_loc <- [loc];
        Buffer.reset state.buffer;
        comment state lexbuf >>= fun end_loc ->
        let s = Buffer.contents state.buffer in
        Buffer.reset state.buffer;
        return (COMMENT (s, { loc with Location.loc_end = end_loc.Location.loc_end }))
      )
# 3104 "src/ocaml/preprocess/lexer_raw.ml"

  | 38 ->
# 533 "src/ocaml/preprocess/lexer_raw.mll"
      ( let loc = Location.curr lexbuf in
        Location.prerr_warning loc Warnings.Comment_not_end;
        lexbuf.Lexing.lex_curr_pos <- lexbuf.Lexing.lex_curr_pos - 1;
        let curpos = lexbuf.lex_curr_p in
        lexbuf.lex_curr_p <- { curpos with pos_cnum = curpos.pos_cnum - 1 };
        return STAR
      )
# 3115 "src/ocaml/preprocess/lexer_raw.ml"

  | 39 ->
let
# 540 "src/ocaml/preprocess/lexer_raw.mll"
                                   num
# 3121 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 541 "src/ocaml/preprocess/lexer_raw.mll"
                                            name
# 3126 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_mem.(2) in
# 543 "src/ocaml/preprocess/lexer_raw.mll"
      ( update_loc lexbuf name (int_of_string num) true 0;
        token state lexbuf
      )
# 3132 "src/ocaml/preprocess/lexer_raw.ml"

  | 40 ->
# 546 "src/ocaml/preprocess/lexer_raw.mll"
         ( return HASH )
# 3137 "src/ocaml/preprocess/lexer_raw.ml"

  | 41 ->
# 547 "src/ocaml/preprocess/lexer_raw.mll"
         ( return AMPERSAND )
# 3142 "src/ocaml/preprocess/lexer_raw.ml"

  | 42 ->
# 548 "src/ocaml/preprocess/lexer_raw.mll"
         ( return AMPERAMPER )
# 3147 "src/ocaml/preprocess/lexer_raw.ml"

  | 43 ->
# 549 "src/ocaml/preprocess/lexer_raw.mll"
         ( return BACKQUOTE )
# 3152 "src/ocaml/preprocess/lexer_raw.ml"

  | 44 ->
# 550 "src/ocaml/preprocess/lexer_raw.mll"
         ( return QUOTE )
# 3157 "src/ocaml/preprocess/lexer_raw.ml"

  | 45 ->
# 551 "src/ocaml/preprocess/lexer_raw.mll"
         ( return LPAREN )
# 3162 "src/ocaml/preprocess/lexer_raw.ml"

  | 46 ->
# 552 "src/ocaml/preprocess/lexer_raw.mll"
         ( return RPAREN )
# 3167 "src/ocaml/preprocess/lexer_raw.ml"

  | 47 ->
# 553 "src/ocaml/preprocess/lexer_raw.mll"
         ( return STAR )
# 3172 "src/ocaml/preprocess/lexer_raw.ml"

  | 48 ->
# 554 "src/ocaml/preprocess/lexer_raw.mll"
         ( return COMMA )
# 3177 "src/ocaml/preprocess/lexer_raw.ml"

  | 49 ->
# 555 "src/ocaml/preprocess/lexer_raw.mll"
         ( return MINUSGREATER )
# 3182 "src/ocaml/preprocess/lexer_raw.ml"

  | 50 ->
# 556 "src/ocaml/preprocess/lexer_raw.mll"
         ( return DOT )
# 3187 "src/ocaml/preprocess/lexer_raw.ml"

  | 51 ->
let
# 557 "src/ocaml/preprocess/lexer_raw.mll"
                                      op
# 3193 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) lexbuf.Lexing.lex_curr_pos in
# 557 "src/ocaml/preprocess/lexer_raw.mll"
                                          ( return (DOTOP op) )
# 3197 "src/ocaml/preprocess/lexer_raw.ml"

  | 52 ->
# 558 "src/ocaml/preprocess/lexer_raw.mll"
         ( return DOTDOT )
# 3202 "src/ocaml/preprocess/lexer_raw.ml"

  | 53 ->
# 559 "src/ocaml/preprocess/lexer_raw.mll"
         ( return COLON )
# 3207 "src/ocaml/preprocess/lexer_raw.ml"

  | 54 ->
# 560 "src/ocaml/preprocess/lexer_raw.mll"
         ( return COLONCOLON )
# 3212 "src/ocaml/preprocess/lexer_raw.ml"

  | 55 ->
# 561 "src/ocaml/preprocess/lexer_raw.mll"
         ( return COLONEQUAL )
# 3217 "src/ocaml/preprocess/lexer_raw.ml"

  | 56 ->
# 562 "src/ocaml/preprocess/lexer_raw.mll"
         ( return COLONGREATER )
# 3222 "src/ocaml/preprocess/lexer_raw.ml"

  | 57 ->
# 563 "src/ocaml/preprocess/lexer_raw.mll"
         ( return SEMI )
# 3227 "src/ocaml/preprocess/lexer_raw.ml"

  | 58 ->
# 564 "src/ocaml/preprocess/lexer_raw.mll"
         ( return SEMISEMI )
# 3232 "src/ocaml/preprocess/lexer_raw.ml"

  | 59 ->
# 565 "src/ocaml/preprocess/lexer_raw.mll"
         ( return LESS )
# 3237 "src/ocaml/preprocess/lexer_raw.ml"

  | 60 ->
# 566 "src/ocaml/preprocess/lexer_raw.mll"
         ( return LESSMINUS )
# 3242 "src/ocaml/preprocess/lexer_raw.ml"

  | 61 ->
# 567 "src/ocaml/preprocess/lexer_raw.mll"
         ( return EQUAL )
# 3247 "src/ocaml/preprocess/lexer_raw.ml"

  | 62 ->
# 568 "src/ocaml/preprocess/lexer_raw.mll"
         ( return LBRACKET )
# 3252 "src/ocaml/preprocess/lexer_raw.ml"

  | 63 ->
# 569 "src/ocaml/preprocess/lexer_raw.mll"
         ( return LBRACKETBAR )
# 3257 "src/ocaml/preprocess/lexer_raw.ml"

  | 64 ->
# 570 "src/ocaml/preprocess/lexer_raw.mll"
         ( return LBRACKETLESS )
# 3262 "src/ocaml/preprocess/lexer_raw.ml"

  | 65 ->
# 571 "src/ocaml/preprocess/lexer_raw.mll"
         ( return LBRACKETGREATER )
# 3267 "src/ocaml/preprocess/lexer_raw.ml"

  | 66 ->
# 572 "src/ocaml/preprocess/lexer_raw.mll"
         ( return RBRACKET )
# 3272 "src/ocaml/preprocess/lexer_raw.ml"

  | 67 ->
# 573 "src/ocaml/preprocess/lexer_raw.mll"
         ( return LBRACE )
# 3277 "src/ocaml/preprocess/lexer_raw.ml"

  | 68 ->
# 574 "src/ocaml/preprocess/lexer_raw.mll"
         ( return LBRACELESS )
# 3282 "src/ocaml/preprocess/lexer_raw.ml"

  | 69 ->
# 575 "src/ocaml/preprocess/lexer_raw.mll"
         ( return BAR )
# 3287 "src/ocaml/preprocess/lexer_raw.ml"

  | 70 ->
# 576 "src/ocaml/preprocess/lexer_raw.mll"
         ( return BARBAR )
# 3292 "src/ocaml/preprocess/lexer_raw.ml"

  | 71 ->
# 577 "src/ocaml/preprocess/lexer_raw.mll"
         ( return BARRBRACKET )
# 3297 "src/ocaml/preprocess/lexer_raw.ml"

  | 72 ->
# 578 "src/ocaml/preprocess/lexer_raw.mll"
         ( return GREATER )
# 3302 "src/ocaml/preprocess/lexer_raw.ml"

  | 73 ->
# 579 "src/ocaml/preprocess/lexer_raw.mll"
         ( return GREATERRBRACKET )
# 3307 "src/ocaml/preprocess/lexer_raw.ml"

  | 74 ->
# 580 "src/ocaml/preprocess/lexer_raw.mll"
         ( return RBRACE )
# 3312 "src/ocaml/preprocess/lexer_raw.ml"

  | 75 ->
# 581 "src/ocaml/preprocess/lexer_raw.mll"
         ( return GREATERRBRACE )
# 3317 "src/ocaml/preprocess/lexer_raw.ml"

  | 76 ->
# 582 "src/ocaml/preprocess/lexer_raw.mll"
         ( return LBRACKETAT )
# 3322 "src/ocaml/preprocess/lexer_raw.ml"

  | 77 ->
# 583 "src/ocaml/preprocess/lexer_raw.mll"
           ( return LBRACKETATAT )
# 3327 "src/ocaml/preprocess/lexer_raw.ml"

  | 78 ->
# 584 "src/ocaml/preprocess/lexer_raw.mll"
           ( return LBRACKETATATAT )
# 3332 "src/ocaml/preprocess/lexer_raw.ml"

  | 79 ->
# 585 "src/ocaml/preprocess/lexer_raw.mll"
         ( return LBRACKETPERCENT )
# 3337 "src/ocaml/preprocess/lexer_raw.ml"

  | 80 ->
# 586 "src/ocaml/preprocess/lexer_raw.mll"
          ( return LBRACKETPERCENTPERCENT )
# 3342 "src/ocaml/preprocess/lexer_raw.ml"

  | 81 ->
# 587 "src/ocaml/preprocess/lexer_raw.mll"
         ( return BANG )
# 3347 "src/ocaml/preprocess/lexer_raw.ml"

  | 82 ->
# 588 "src/ocaml/preprocess/lexer_raw.mll"
         ( return (INFIXOP0 "!=") )
# 3352 "src/ocaml/preprocess/lexer_raw.ml"

  | 83 ->
# 589 "src/ocaml/preprocess/lexer_raw.mll"
         ( return PLUS )
# 3357 "src/ocaml/preprocess/lexer_raw.ml"

  | 84 ->
# 590 "src/ocaml/preprocess/lexer_raw.mll"
         ( return PLUSDOT )
# 3362 "src/ocaml/preprocess/lexer_raw.ml"

  | 85 ->
# 591 "src/ocaml/preprocess/lexer_raw.mll"
         ( return PLUSEQ )
# 3367 "src/ocaml/preprocess/lexer_raw.ml"

  | 86 ->
# 592 "src/ocaml/preprocess/lexer_raw.mll"
         ( return MINUS )
# 3372 "src/ocaml/preprocess/lexer_raw.ml"

  | 87 ->
# 593 "src/ocaml/preprocess/lexer_raw.mll"
         ( return MINUSDOT )
# 3377 "src/ocaml/preprocess/lexer_raw.ml"

  | 88 ->
let
# 595 "src/ocaml/preprocess/lexer_raw.mll"
                                op
# 3383 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 596 "src/ocaml/preprocess/lexer_raw.mll"
            ( return (PREFIXOP op) )
# 3387 "src/ocaml/preprocess/lexer_raw.ml"

  | 89 ->
let
# 597 "src/ocaml/preprocess/lexer_raw.mll"
                                      op
# 3393 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 598 "src/ocaml/preprocess/lexer_raw.mll"
            ( return (PREFIXOP op) )
# 3397 "src/ocaml/preprocess/lexer_raw.ml"

  | 90 ->
let
# 599 "src/ocaml/preprocess/lexer_raw.mll"
                                              op
# 3403 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 600 "src/ocaml/preprocess/lexer_raw.mll"
            ( return (keyword_or state op
                       (INFIXOP0 op)) )
# 3408 "src/ocaml/preprocess/lexer_raw.ml"

  | 91 ->
let
# 602 "src/ocaml/preprocess/lexer_raw.mll"
                              op
# 3414 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 603 "src/ocaml/preprocess/lexer_raw.mll"
            ( return (INFIXOP1 op) )
# 3418 "src/ocaml/preprocess/lexer_raw.ml"

  | 92 ->
let
# 604 "src/ocaml/preprocess/lexer_raw.mll"
                              op
# 3424 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 605 "src/ocaml/preprocess/lexer_raw.mll"
            ( return (INFIXOP2 op) )
# 3428 "src/ocaml/preprocess/lexer_raw.ml"

  | 93 ->
let
# 606 "src/ocaml/preprocess/lexer_raw.mll"
                         op
# 3434 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 607 "src/ocaml/preprocess/lexer_raw.mll"
            ( return (INFIXOP4 op) )
# 3438 "src/ocaml/preprocess/lexer_raw.ml"

  | 94 ->
# 608 "src/ocaml/preprocess/lexer_raw.mll"
            ( return PERCENT )
# 3443 "src/ocaml/preprocess/lexer_raw.ml"

  | 95 ->
let
# 609 "src/ocaml/preprocess/lexer_raw.mll"
                                  op
# 3449 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 610 "src/ocaml/preprocess/lexer_raw.mll"
            ( return (INFIXOP3 op) )
# 3453 "src/ocaml/preprocess/lexer_raw.ml"

  | 96 ->
let
# 612 "src/ocaml/preprocess/lexer_raw.mll"
                                op
# 3459 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 613 "src/ocaml/preprocess/lexer_raw.mll"
            ( return (try Hashtbl.find state.keywords op
                      with Not_found -> HASHOP op) )
# 3464 "src/ocaml/preprocess/lexer_raw.ml"

  | 97 ->
let
# 615 "src/ocaml/preprocess/lexer_raw.mll"
                                       op
# 3470 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 616 "src/ocaml/preprocess/lexer_raw.mll"
            ( return (LETOP op) )
# 3474 "src/ocaml/preprocess/lexer_raw.ml"

  | 98 ->
let
# 617 "src/ocaml/preprocess/lexer_raw.mll"
                                       op
# 3480 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 618 "src/ocaml/preprocess/lexer_raw.mll"
            ( return (ANDOP op) )
# 3484 "src/ocaml/preprocess/lexer_raw.ml"

  | 99 ->
# 619 "src/ocaml/preprocess/lexer_raw.mll"
        ( return EOF )
# 3489 "src/ocaml/preprocess/lexer_raw.ml"

  | 100 ->
let
# 621 "src/ocaml/preprocess/lexer_raw.mll"
         illegal_char
# 3495 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 622 "src/ocaml/preprocess/lexer_raw.mll"
      ( fail lexbuf (Illegal_character illegal_char) )
# 3499 "src/ocaml/preprocess/lexer_raw.ml"

  | __ocaml_lex_state -> __ocaml_lex_refill
      (fun lexbuf -> lexbuf.Lexing.refill_buff lexbuf;
         __ocaml_lex_token_rec state lexbuf __ocaml_lex_state) lexbuf

and comment state lexbuf =
   __ocaml_lex_comment_rec state lexbuf 168
and __ocaml_lex_comment_rec state lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 626 "src/ocaml/preprocess/lexer_raw.mll"
      ( state.comment_start_loc <- (Location.curr lexbuf) :: state.comment_start_loc;
      Buffer.add_string state.buffer (Lexing.lexeme lexbuf);
      comment state lexbuf
    )
# 3515 "src/ocaml/preprocess/lexer_raw.ml"

  | 1 ->
# 631 "src/ocaml/preprocess/lexer_raw.mll"
      ( match state.comment_start_loc with
        | [] -> assert false
        | [_] -> state.comment_start_loc <- []; return (Location.curr lexbuf)
        | _ :: l -> state.comment_start_loc <- l;
                  Buffer.add_string state.buffer (Lexing.lexeme lexbuf);
                  comment state lexbuf
       )
# 3526 "src/ocaml/preprocess/lexer_raw.ml"

  | 2 ->
# 639 "src/ocaml/preprocess/lexer_raw.mll"
      (
        state.string_start_loc <- Location.curr lexbuf;
        Buffer.add_char state.buffer '\"';
        let buffer = state.buffer in
        state.buffer <- Buffer.create 15;
        (catch (string state lexbuf) (fun e l -> match e with
             | Unterminated_string ->
               begin match state.comment_start_loc with
                 | [] -> assert false
                 | loc :: _ ->
                   let start = List.hd (List.rev state.comment_start_loc) in
                   state.comment_start_loc <- [];
                   fail_loc (Unterminated_string_in_comment (start, l)) loc
               end
             | e -> fail_loc e l
           )
        ) >>= fun _loc ->
      state.string_start_loc <- Location.none;
      Buffer.add_string buffer (String.escaped (Buffer.contents state.buffer));
      state.buffer <- buffer;
      Buffer.add_char state.buffer '\"';
      comment state lexbuf )
# 3552 "src/ocaml/preprocess/lexer_raw.ml"

  | 3 ->
# 662 "src/ocaml/preprocess/lexer_raw.mll"
      (
        let delim = Lexing.lexeme lexbuf in
        let delim = String.sub delim ~pos:1 ~len:(String.length delim - 2) in
        state.string_start_loc <- Location.curr lexbuf;
        Buffer.add_string state.buffer (Lexing.lexeme lexbuf);
        (catch (quoted_string delim state lexbuf) (fun e l -> match e with
             | Unterminated_string ->
               begin match state.comment_start_loc with
                 | [] -> assert false
                 | loc :: _ ->
                   let start = List.hd (List.rev state.comment_start_loc) in
                   state.comment_start_loc <- [];
                   fail_loc (Unterminated_string_in_comment (start, l)) loc
               end
             | e -> fail_loc e l
           )
        ) >>= fun _loc ->
        state.string_start_loc <- Location.none;
        Buffer.add_char state.buffer '|';
        Buffer.add_string state.buffer delim;
        Buffer.add_char state.buffer '}';
        comment state lexbuf )
# 3578 "src/ocaml/preprocess/lexer_raw.ml"

  | 4 ->
# 686 "src/ocaml/preprocess/lexer_raw.mll"
      ( Buffer.add_string state.buffer (Lexing.lexeme lexbuf); comment state lexbuf )
# 3583 "src/ocaml/preprocess/lexer_raw.ml"

  | 5 ->
# 688 "src/ocaml/preprocess/lexer_raw.mll"
      ( update_loc lexbuf None 1 false 1;
        Buffer.add_string state.buffer (Lexing.lexeme lexbuf);
        comment state lexbuf
      )
# 3591 "src/ocaml/preprocess/lexer_raw.ml"

  | 6 ->
# 693 "src/ocaml/preprocess/lexer_raw.mll"
      ( Buffer.add_string state.buffer (Lexing.lexeme lexbuf); comment state lexbuf )
# 3596 "src/ocaml/preprocess/lexer_raw.ml"

  | 7 ->
# 695 "src/ocaml/preprocess/lexer_raw.mll"
      ( Buffer.add_string state.buffer (Lexing.lexeme lexbuf); comment state lexbuf )
# 3601 "src/ocaml/preprocess/lexer_raw.ml"

  | 8 ->
# 697 "src/ocaml/preprocess/lexer_raw.mll"
      ( Buffer.add_string state.buffer (Lexing.lexeme lexbuf); comment state lexbuf )
# 3606 "src/ocaml/preprocess/lexer_raw.ml"

  | 9 ->
# 699 "src/ocaml/preprocess/lexer_raw.mll"
      ( Buffer.add_string state.buffer (Lexing.lexeme lexbuf); comment state lexbuf )
# 3611 "src/ocaml/preprocess/lexer_raw.ml"

  | 10 ->
# 701 "src/ocaml/preprocess/lexer_raw.mll"
      ( Buffer.add_string state.buffer (Lexing.lexeme lexbuf); comment state lexbuf )
# 3616 "src/ocaml/preprocess/lexer_raw.ml"

  | 11 ->
# 703 "src/ocaml/preprocess/lexer_raw.mll"
      ( match state.comment_start_loc with
        | [] -> assert false
        | loc :: _ ->
          let start = List.hd (List.rev state.comment_start_loc) in
          state.comment_start_loc <- [];
          fail_loc (Unterminated_comment start) loc
      )
# 3627 "src/ocaml/preprocess/lexer_raw.ml"

  | 12 ->
# 711 "src/ocaml/preprocess/lexer_raw.mll"
      ( update_loc lexbuf None 1 false 0;
        Buffer.add_string state.buffer (Lexing.lexeme lexbuf);
        comment state lexbuf
      )
# 3635 "src/ocaml/preprocess/lexer_raw.ml"

  | 13 ->
# 716 "src/ocaml/preprocess/lexer_raw.mll"
      ( Buffer.add_string state.buffer (Lexing.lexeme lexbuf); comment state lexbuf )
# 3640 "src/ocaml/preprocess/lexer_raw.ml"

  | 14 ->
# 718 "src/ocaml/preprocess/lexer_raw.mll"
      ( Buffer.add_string state.buffer (Lexing.lexeme lexbuf); comment state lexbuf )
# 3645 "src/ocaml/preprocess/lexer_raw.ml"

  | __ocaml_lex_state -> __ocaml_lex_refill
      (fun lexbuf -> lexbuf.Lexing.refill_buff lexbuf;
         __ocaml_lex_comment_rec state lexbuf __ocaml_lex_state) lexbuf

and string state lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 2 (-1); __ocaml_lex_string_rec state lexbuf 212
and __ocaml_lex_string_rec state lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 722 "src/ocaml/preprocess/lexer_raw.mll"
      ( return lexbuf.lex_start_p  )
# 3658 "src/ocaml/preprocess/lexer_raw.ml"

  | 1 ->
let
# 723 "src/ocaml/preprocess/lexer_raw.mll"
                                  space
# 3664 "src/ocaml/preprocess/lexer_raw.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 724 "src/ocaml/preprocess/lexer_raw.mll"
      ( update_loc lexbuf None 1 false (String.length space);
        string state lexbuf
      )
# 3670 "src/ocaml/preprocess/lexer_raw.ml"

  | 2 ->
# 728 "src/ocaml/preprocess/lexer_raw.mll"
      ( Buffer.add_char state.buffer
          (char_for_backslash (Lexing.lexeme_char lexbuf 1));
        string state lexbuf )
# 3677 "src/ocaml/preprocess/lexer_raw.ml"

  | 3 ->
# 732 "src/ocaml/preprocess/lexer_raw.mll"
      ( char_for_decimal_code state lexbuf 1 >>= fun c ->
        Buffer.add_char state.buffer c;
        string state lexbuf )
# 3684 "src/ocaml/preprocess/lexer_raw.ml"

  | 4 ->
# 736 "src/ocaml/preprocess/lexer_raw.mll"
      ( Buffer.add_char state.buffer (char_for_hexadecimal_code lexbuf 2);
        string state lexbuf )
# 3690 "src/ocaml/preprocess/lexer_raw.ml"

  | 5 ->
# 739 "src/ocaml/preprocess/lexer_raw.mll"
      ( store_escaped_uchar state lexbuf (uchar_for_uchar_escape lexbuf);
        string state lexbuf )
# 3696 "src/ocaml/preprocess/lexer_raw.ml"

  | 6 ->
# 742 "src/ocaml/preprocess/lexer_raw.mll"
      ( if in_comment state
        then string state lexbuf
        else begin
(*  Should be an error, but we are very lax.
                  fail (Illegal_escape (Lexing.lexeme lexbuf),
                        (Location.curr lexbuf)
*)
          let loc = Location.curr lexbuf in
          Location.prerr_warning loc Warnings.Illegal_backslash;
          Buffer.add_char state.buffer (Lexing.lexeme_char lexbuf 0);
          Buffer.add_char state.buffer (Lexing.lexeme_char lexbuf 1);
          string state lexbuf
        end
      )
# 3714 "src/ocaml/preprocess/lexer_raw.ml"

  | 7 ->
# 757 "src/ocaml/preprocess/lexer_raw.mll"
      ( if not (in_comment state) then
          Location.prerr_warning (Location.curr lexbuf) Warnings.Eol_in_string;
        update_loc lexbuf None 1 false 0;
        Buffer.add_string state.buffer (Lexing.lexeme lexbuf);
        string state lexbuf
      )
# 3724 "src/ocaml/preprocess/lexer_raw.ml"

  | 8 ->
# 764 "src/ocaml/preprocess/lexer_raw.mll"
      ( let loc = state.string_start_loc in
        state.string_start_loc <- Location.none;
        fail_loc Unterminated_string loc )
# 3731 "src/ocaml/preprocess/lexer_raw.ml"

  | 9 ->
# 768 "src/ocaml/preprocess/lexer_raw.mll"
      ( Buffer.add_char state.buffer (Lexing.lexeme_char lexbuf 0);
        string state lexbuf )
# 3737 "src/ocaml/preprocess/lexer_raw.ml"

  | __ocaml_lex_state -> __ocaml_lex_refill
      (fun lexbuf -> lexbuf.Lexing.refill_buff lexbuf;
         __ocaml_lex_string_rec state lexbuf __ocaml_lex_state) lexbuf

and quoted_string delim state lexbuf =
   __ocaml_lex_quoted_string_rec delim state lexbuf 235
and __ocaml_lex_quoted_string_rec delim state lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 773 "src/ocaml/preprocess/lexer_raw.mll"
      ( update_loc lexbuf None 1 false 0;
        Buffer.add_string state.buffer (Lexing.lexeme lexbuf);
        quoted_string delim state lexbuf
      )
# 3753 "src/ocaml/preprocess/lexer_raw.ml"

  | 1 ->
# 778 "src/ocaml/preprocess/lexer_raw.mll"
      ( let loc = state.string_start_loc in
        state.string_start_loc <- Location.none;
        fail_loc Unterminated_string loc )
# 3760 "src/ocaml/preprocess/lexer_raw.ml"

  | 2 ->
# 782 "src/ocaml/preprocess/lexer_raw.mll"
      (
        let edelim = Lexing.lexeme lexbuf in
        let edelim = String.sub edelim ~pos:1 ~len:(String.length edelim - 2) in
        if delim = edelim then return lexbuf.lex_start_p
        else (Buffer.add_string state.buffer (Lexing.lexeme lexbuf);
              quoted_string delim state lexbuf)
      )
# 3771 "src/ocaml/preprocess/lexer_raw.ml"

  | 3 ->
# 790 "src/ocaml/preprocess/lexer_raw.mll"
      ( Buffer.add_char state.buffer (Lexing.lexeme_char lexbuf 0);
        quoted_string delim state lexbuf )
# 3777 "src/ocaml/preprocess/lexer_raw.ml"

  | __ocaml_lex_state -> __ocaml_lex_refill
      (fun lexbuf -> lexbuf.Lexing.refill_buff lexbuf;
         __ocaml_lex_quoted_string_rec delim state lexbuf __ocaml_lex_state) lexbuf

and skip_sharp_bang state lexbuf =
   __ocaml_lex_skip_sharp_bang_rec state lexbuf 244
and __ocaml_lex_skip_sharp_bang_rec state lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 795 "src/ocaml/preprocess/lexer_raw.mll"
      ( update_loc lexbuf None 3 false 0; token state lexbuf )
# 3790 "src/ocaml/preprocess/lexer_raw.ml"

  | 1 ->
# 797 "src/ocaml/preprocess/lexer_raw.mll"
      ( update_loc lexbuf None 1 false 0; token state lexbuf )
# 3795 "src/ocaml/preprocess/lexer_raw.ml"

  | 2 ->
# 798 "src/ocaml/preprocess/lexer_raw.mll"
       ( token state lexbuf )
# 3800 "src/ocaml/preprocess/lexer_raw.ml"

  | __ocaml_lex_state -> __ocaml_lex_refill
      (fun lexbuf -> lexbuf.Lexing.refill_buff lexbuf;
         __ocaml_lex_skip_sharp_bang_rec state lexbuf __ocaml_lex_state) lexbuf

;;

# 800 "src/ocaml/preprocess/lexer_raw.mll"
 
  type comment = string * Location.t

  (* preprocessor support not implemented, not compatible with monadic
     interface *)

  let rec token_without_comments state lexbuf =
    token state lexbuf >>= function
    | COMMENT _ ->
      token_without_comments state lexbuf
    | tok -> return tok

# 3821 "src/ocaml/preprocess/lexer_raw.ml"
