__version__   = "$Revision: 1.3 $"[11:-2]
__copyright__ = """Copyright (c) 2003 Not Another Corporation Incorporated
                   www.notanothercorporation.com"""
__license__   = """Licensed under the GNU LGPL

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""
__doc__ = """
** %(name)s **
File version %(version)s::
 
    %(copyright)s

    %(license)s

Mongoose
========

Main part of the C{mongoose} package, provides routines to:

  - Hook and unhook exception handler.
  - Register reporters.
  - Set session information.

Usage example::

   import mongoose


   mongoose.start()

   mongoose.registerReporter(WindowReporter(email='foo@nac.com'))

     < ... code here ... >

   mongoose.stop()

Where C{WindowReporter} is a sub-class of C{mongoose.Reporter}.

The C{mongoose.start()} can also go after C{mongoose.registerReporter()} but
exceptions from the reporter registration won't be caught.

I{* $Id: mongoose.py,v 1.3 2003/09/24 17:03:31 philiplindsay Exp $ *}
""" % {'name':__name__, 'version':__version__, 'copyright':__copyright__,
       'license':__license__}

import sys
import time

#import mongoose # TODO: Just move this into here?
from sessioninfo import SessionInfo
from unhandledexception import UnhandledException, calcIncidentId 

try:
    True
except NameError:
    True = (1 == 1)
    False = not True
                 
_DEBUG = False

def setDebug(mode):
    """
    Controls whether or not we act as if we're being debugged.
    """
    global _DEBUG

    _DEBUG = mode


class Reporter:
    """
    Abstract class to be overridden by all I{reporters} that want to be alerted
    when an unhandled exception occurs.

    NOTE: The routines in this class should I{not} propagate
    exceptions that occur during their execution, any exceptions
    propagated will be ignored by the calling controller.
    """

    def processAndUnload(self, unhandledException, sessionInfo):
        """
        Called by the reporter controller when it has caught an
        otherwise unhandled exception.

        Currently ensures the reporter is only ever called once by forcing
        a call C{unload()} after a call to C{process()}

        @param unhandledException: An C{UnhandledException} instance.
        @param sessionInfo: A C{SessionInfo} instance.
        """
        self.process(unhandledException, sessionInfo)
        self.unload()


    def process(self, unhandledException, sessionInfo):
        """
        Called when an unhandled exception occurs.

        This method must be overridden by a method that reports the
        unhandled exception in some manner.

        Currently this routine will only ever be called once, as part of the
        sequence of calls started by a call to C{processAndUnload}.

        NOTE: This routine should I{not} propagate exceptions that occur
        during its execution, any exceptions propagated will be ignored
        by the calling controller.

        @param unhandledException: An C{UnhandledException} instance.
        @param sessionInfo: A C{SessionInfo} instance.
        """
        pass

        
    def unload(self):
        """
        Called by the reporter controller when it has been instructed
        to stop.  At present this method is also always called after every
        call to C{process()}.

        This method should do any clean up the reporter needs to do
        before it ceases existing. (e.g. release resources)

        NOTE: This routine should I{not} propagate exceptions that occur
        during its execution, any exceptions propagated will be ignored
        by the calling controller.
        """
        pass


_reporters = []
def registerReporter(newReporter):
    """
    Registers the supplied reporter so it is notified when an unhandled
    exception occurs.

    Reporters are registered on a first-in, first-called basis.
    """
    _reporters.append(newReporter)


def unregisterReporters():
    """
    Unregisters all registered reporters by calling their C{unload()} method
    and then deleting them.

    NOTE: This routine I{ignores} exceptions propagated by the reporters.
    """
    while len(_reporters):
        try:
            _reporters[-1].unload()
        except:
            if _DEBUG:
                raise

        del _reporters[-1]
    

def start():
    """
    Replaces the current system exception handler with our own.
    """
    sys.excepthook = _exceptionHook

    
def stop():
    """
    Returns the system exception handler to the default and unloads registered
    reporters.

    NOTE: This may I{not} be the handler that was installed before ours.
    """
    sys.excepthook = sys.__excepthook__

    unregisterReporters()


_sessionInfo = {}
def setSessionInfo(name, value):
    """
    Sets an attribute of our session information. The value is always
    stringified.

    If an exception occurs it is ignored.
    """
    try:
        _sessionInfo[name] = str(value)
    except:
        if _DEBUG:
            raise


def _exceptionHook(exctype, value, traceback):
    """
    A replacement system exception handler.

    If we're not in debug mode:
      - We call the reporters.
          - If a reporter throws an exception we ignore it.
      - We exit the system.

    If we're in debug mode:
      - We call the reporters.
          - If a reporter throws an exception we re-throw it.
      - Then we throw the exception.

    @param exctype:
    @param value:
    @param traceback:
    """
    try:
        # Attempt to get values for our exception information.
        try:
            incidentTime = time.asctime()
        except:
            if _DEBUG:
                raise
            else:
                incidentTime = ""

        # Create the objects...
        # TODO: Should we wrap these in try/except as well?
        uhx = UnhandledException(exctype,
                                 value, traceback, incidentTime, _DEBUG)
        
        info = SessionInfo(_sessionInfo)

        # Notify all the registered reporters...
        while len(_reporters):
            try:
                _reporters[0].processAndUnload(uhx, info)
            except:
                if _DEBUG:
                    raise
                
            del _reporters[0] # The postman only knocks once...

    except:
        # We've had an exception in our handler...
        if _DEBUG:
            raise
        else:
            #...let's just give up!
            sys.exit(1)

    if _DEBUG:
        sys.__excepthook__(exctype, value, traceback)
    else:
        sys.exit(1)
            

