# -*- coding: utf-8 -*-
#   Copyright (C) 2009-2010, 2013, 2015, 2020, 2024 Rocky Bernstein
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

from trepan.lib.complete import complete_token
from trepan.processor.command.base_cmd import DebuggerCommand


class UndisplayCommand(DebuggerCommand):
    """**undisplay** *display-number*...

    Cancel some expressions to be displayed when program stops.
    Arguments are the code numbers of the expressions to stop displaying.

    No argument cancels all automatic-display expressions and is
    the same as `delete display`.

    See also:
    ---------

    `info display` to see current list of code numbers."""

    aliases = ("und",)
    short_help = "Cancel some expressions to be displayed " "when program stops"

    DebuggerCommand.setup(locals(), category="data", min_args=1)

    def complete(self, prefix):
        completions = [str(disp.number) for disp in self.proc.display_mgr.list]
        return complete_token(completions, prefix)

    def run(self, args):
        if len(args) == 1:
            self.proc.display_mgr.clear()
            return
        for i in args[1:]:
            i = self.proc.get_an_int(i, "%r must be a display number" % i)
            if i is not None:
                if not self.proc.display_mgr.delete_index(i):
                    self.errmsg("No display number %d." % i)
                    return
                pass
            pass
        return False

    pass


if __name__ == "__main__":
    import inspect

    from trepan.debugger import Trepan
    from trepan.processor.cmdproc import get_stack

    d = Trepan()
    cp = d.core.processor
    if cp is not None:
        command = UndisplayCommand(d.core.processor)
        cp.curframe = inspect.currentframe()
        if cp.curframe is None:
            print("Can't get current frame - no demo done")
        else:
            cp.stack, cp.curindex = get_stack(cp.curframe, None, None, cp)
            command.run(["undisplay", "z"])
            command.run(["undisplay", "1", "10"])
        pass
    pass
